<?php
/**
 * MTS Car Booking 料金表管理のビュー
 *
 * @Filename    ChargeAdminView.php
 * @Author      S.Hayashi
 * @Code        2018-07-07 Ver.1.0.0
 */
namespace MTSCarBookingTrial\views;

use MTSCarBookingTrial\Config;
use MTSCarBookingTrial\CustomPost;
use MTSCarBookingTrial\ChargeAdmin;
use MTSCarBookingTrial\models\Charge;


class ChargeAdminView
{
    const PAGE_NAME = ChargeAdmin::PAGE_NAME;

    const JS = 'js/charge-admin.js';
    const CSS = '/css/mtsrcb-admin.css';

    private $oHelper = null;

    private $oCharge = null;

    /**
     * Constructor
     */
    public function __construct(Charge $oCharge)
    {
        $this->oHelper = FormHelper::getInstance();
        $this->oCharge = $oCharge;

        // JavaScript、CSSソースの読み込み指定
        add_action('admin_enqueue_scripts', array($this, 'enqueueScript'));

        // 管理画面追加編集ブロック登録
        $idname = self::PAGE_NAME . '-';
        add_meta_box("{$idname}name", '料金表名称', array($this, 'nameBox'), CustomPost::CHARGE, 'normal', 'default', array('oCharge' => $oCharge));
        add_meta_box("{$idname}charge", 'レンタル料金', array($this, 'chargeBox'), CustomPost::CHARGE, 'normal', 'default', array('oCharge' => $oCharge));
        add_meta_box("{$idname}regular", '料金表', array($this, 'regularBox'), CustomPost::CHARGE, 'normal', 'default', array('oCharge' => $oCharge));
        add_meta_box("{$idname}limited", '特別料金表', array($this, 'limitedBox'), CustomPost::CHARGE, 'normal', 'default', array('oCharge' => $oCharge));
        add_meta_box("{$idname}period", '特別料金期間限定', array($this, 'periodBox'), CustomPost::CHARGE, 'normal', 'default', array('oCharge' => $oCharge));

        // エラー表示登録
        add_action('admin_notices', array($this, 'errorBox'));
    }

    /**
     * JavaScript,Style headタグ内のロード登録
     */
    public function enqueueScript()
    {
        $dir = dirname(__FILE__);

        wp_enqueue_script(self::PAGE_NAME, plugins_url(self::JS, $dir), array('jquery'));
        wp_enqueue_style(self::PAGE_NAME, plugins_url(self::CSS, $dir));
    }

    /**
     * 料金表名称
     */
    public function nameBox($oPost, $meta)
    {
        $oCharge = $meta['args']['oCharge'];

        foreach (Config::$languages as $lang => $title) {
            echo '<div class="line-box">';
            echo sprintf('<label for="name-%s" class="lang-title">%s</label>', $lang, $title);
            //echo sprintf('<input id="name-%s" class="mts-middle" type="text" name="vehicle[name][%s]" value="%s">', $lang, $lang, $oVehicle->name->$lang);
            echo $this->oHelper->textBox("name-{$lang}", 'mts-middle', "charge[name][{$lang}]", (property_exists($oCharge->name, $lang) ? $oCharge->name->$lang : ''));
            echo "</div>\n";
        }

        echo sprintf('<input type="hidden" name="charge_nonce" value="%s">', wp_create_nonce(self::PAGE_NAME));
    }

    /**
     * 取扱単位
     */
    public function chargeBox($oPost, $meta)
    {
        $oCharge = $meta['args']['oCharge'];

?>
        <table class="mts-form-table">
            <tr>
                <th>対象</th>
                <td>
                    <?php echo $this->oHelper->selectBox('charge-item', '', 'charge[item]', ParamAdmin::$chargeItem, $oCharge->item) ?>
                </td>
            </tr>
            <tr>
                <th>通貨</th>
                <td>
                    <?php echo $this->oHelper->selectBox('charge-currency', '', 'charge[charge][currency]', ParamAdmin::$currency, $oCharge->charge->currency) ?>
                </td>
            </tr>
            <tr>
                <th>単価料金</th>
                <td class="mts-v-middle">
                    <?php $this->_outTimeUnit($oCharge->charge, 'regular') ?>
                </td>
            </tr>
            <tr>
                <th>特別単価料金</th>
                <td class="mts-v-middle">
                    <?php $this->_outTimeUnit($oCharge->charge, 'limited') ?>
                </td>
            </tr>
            <tr>
                <th>計算方法</th>
                <td class="radio-label">
                    <?php echo $this->oHelper->radioButton('charge-method', '', 'charge[charge][method]', ParamAdmin::$compMethod, $oCharge->charge->method) ?>
                    <?php echo $this->oHelper->description('「定額」は期間に関係なく単価料金<br>「単価計算」は時間(日)を１単位とした単価の乗算<br>「表引き」はレンタル期間による表引き金額、超過分は単価計算して加算') ?>
                </td>
            </tr>

        </table>

<?php
    }

    // 単価料金
    private function _outTimeUnit($charge, $tanka)
    {
        $time = $charge->$tanka['time'];

        // 単価単位時間
        if (86400 <= $time) {
            $time = intval($time / 86400);
            $unit = 'day';
        } else {
            $time = intval($time / 3600);
            $unit = 'hour';
        }

        echo $this->oHelper->textBox('', 'mts-center mts-char4', "charge[charge][{$tanka}][time]", $time);
        echo $this->oHelper->selectBox('', 'charge-unit-time', "charge[charge][{$tanka}][unit]", ParamAdmin::$timeUnit, $unit);
        echo $this->oHelper->textBox('', 'mts-small mts-right', "charge[charge][{$tanka}][fee]", $charge->$tanka['fee']);
        echo sprintf('<span class="charge-currency">%s</span>', ParamAdmin::$currency[$charge->currency]);
    }

    /**
     * 通常料金表編集
     */
    public function regularBox($oPost, $meta)
    {
        $oCharge = $meta['args']['oCharge'];

        $this->_outChargeOpe($oCharge, 'regular');

        $this->_outChargeTable($oCharge, 'regular');

?>
        <table id="charge-list-template" style="display:none">
            <tr>
                <td class="charge-table-time"></td>
                <td class="charge-table-price"></td>
                <td class="charge-table-action">
                    <input type="button" value="削除">
                </td>
            </tr>
        </table>

<?php
    }

    // 料金追加操作ライン出力
    private function _outChargeOpe(Charge $oCharge, $type)
    {

?>
        <div id="charge-<?php echo $type ?>" class="ope-line">
            <input type="text" class="charge-time mts-center mts-char4">
            <?php echo $this->oHelper->selectBox('', 'charge-time-unit', '', ParamAdmin::$timeUnit, 'hour') ?>
            <input type="text" class="charge-price mts-small mts-right">
            <span class="charge-currency"><?php echo ParamAdmin::$currency[$oCharge->charge->currency] ?></span>
            <input type="button" class="button" value="追加" onclick="mtsrcb_charge_admin.addCharge('<?php echo $type ?>')">
            <?php echo sprintf('<input id="charge-val-%s" type="hidden" name="charge[%s]" value="%s">',
                $type, $type, esc_html(json_encode($oCharge->$type))) ?>
        </div>
<?php
    }

    // 料金表出力
    private function _outChargeTable(Charge $oCharge, $type)
    {

        $id = sprintf('charge-list-%s', $type);
?>
        <table id="<?php echo $id ?>" class="charge-table">
            <thead>
                <tr>
                    <th class="charge-table-time">時間</th>
                    <th class="charge-table-price">料金</th>
                    <th class="charge-table-action">処理</th>
                </tr>
            </thead>
            <tbody>
            </tbody>
        </table>

<?php
    }

    /**
     * 特別料金表編集
     */
    public function limitedBox($oPost, $meta)
    {
        $oCharge = $meta['args']['oCharge'];

        $this->_outChargeOpe($oCharge, 'limited');

        $this->_outChargeTable($oCharge, 'limited');
    }

    /**
     * 限定期間編集
     */
    public function periodBox($oPost, $meta)
    {
        $oCharge = $meta['args']['oCharge'];

        $start = $end = array('', '', '');
        if (0 < $oCharge->period->start) {
            $start = explode('-', date_i18n('Y-n-j', $oCharge->period->start));
        }
        if (0 < $oCharge->period->end) {
            $end = explode('-', date_i18n('Y-n-j', $oCharge->period->end));
        }

        echo '<div class="ope-line">';
        echo $this->oHelper->textBox('', 'mts-char4', 'charge[period][start][year]', $start[0], '年');
        echo $this->oHelper->selectMonth('', '', 'charge[period][start]', (int) $start[1]);
        echo $this->oHelper->selectDay('', '', 'charge[period][start]', (int) $start[2]);
        echo "\n 〜 ";
        echo $this->oHelper->textBox('', 'mts-char4', 'charge[period][end][year]', $end[0], '年');
        echo $this->oHelper->selectMonth('', '', 'charge[period][end]', (int) $end[1]);
        echo $this->oHelper->selectDay('', '', 'charge[period][end]', (int) $end[2]);
        echo "</div>\n";
    }

    /**
     * エラー表示
     */
    public function errorBox()
    {
        if ($this->oCharge->errCode == '') {
            return;
        }

        echo sprintf('<div class="%s"><p><strong>%s: %s</strong></p></div>',
            'notice-error', $this->oCharge->errCode, $this->oCharge->errSub);
    }

}
