<?php
/**
 * MTS Car Booking 営業カレンダー管理のビュー
 *
 * @Filename    CalendarAdminView.php
 * @Author      S.Hayashi
 * @Code        2018-07-30 Ver.1.0.0
 */
namespace MTSCarBookingTrial\views;

//use MTSCarBookingTrial\CarBooking;
use MTSCarBookingTrial\Config;
use MTSCarBookingTrial\CalendarAdmin;
use MTSCarBookingTrial\models\ShopCalendar;
use MTSCarBookingTrial\models\Calendar;
//use MTSAcbBooking\DataResource;
//use MTSAcbBooking\models\ReserveCar;

class CalendarAdminView
{
    const PAGE_NAME = CalendarAdmin::PAGE_NAME;

    const JS = 'js/calendar-admin.js';
    const CSS = 'css/mtsrcb-admin.css';
    const UI_CSS = 'https://code.jquery.com/ui/1.12.1/themes/smoothness/jquery-ui.css';
    const CALENDAR_ID = 'mtsab-booking-calendar';

    // 管理画面営業カレンダー編集ページ
    private $shopCalendarUrl = '';

    // メッセージ表示データ
    private $messageNo = 0;
    private $message = '';

    private $oHelper = null;

    private $oShopCalendar = null;

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->oHelper = FormHelper::getInstance();

        // 営業カレンダー編集ページのURL
        $this->shopCalendarUrl = admin_url('admin.php?page=' . self::PAGE_NAME);

        // 予約登録・編集ページのURL
        //$this->bookingUrl = admin_url('admin.php?page=' . AcbBooking::BOOKING_ADMIN);

        // JavaScript 組み込み登録の設定
        add_action('admin_enqueue_scripts', array($this, 'enqueueScripts'));
    }

    /**
     * JavaScript,Style headタグ内のロード登録
     */
    public function enqueueScripts()
    {
        $dir = dirname(__FILE__);

        wp_enqueue_script(self::PAGE_NAME, plugins_url(self::JS, $dir), array('jquery', 'jquery-ui-dialog'));
        wp_enqueue_style(self::PAGE_NAME, plugins_url(self::CSS, $dir));
        wp_enqueue_style(self::PAGE_NAME . '-ui', self::UI_CSS);
    }

    /**
     * 営業カレンダー編集画面の表示
     */
    public function editView(ShopCalendar $oShopCalendar, $holidays)
    {
        $this->oShopCalendar = $oShopCalendar;

        ob_start();
?>
        <div class="wrap">
            <?php if ($this->messageNo != 0) {
                $this->_outMessage();
            } ?>

            <h1 class="wp-heading-inline">営業カレンダー編集</h1>

            <h2 class="shop-calendar-title"><?php echo $oShopCalendar->year ?>年</h2>

            <div class="admin-operation">
                <?php $this->_operation() ?>
            </div>

            <div class="calendar-year">
                <?php $this->_calendarAll() ?>
            </div>
            <div class="clear"></div>

            <form method="post">
                <input type="submit" class="button-primary" name="save_calendar" onclick="return mtsrcb_calendar_admin.setCalendarData()" value="保存する">
                <input type="hidden" name="calendar_admin_nonce" value="<?php echo wp_create_nonce(self::PAGE_NAME) ?>">
                <input id="shop_calendar_data" type="hidden" name="shop_calendar" value="">
            </form>

            <div id="shop-day-dialog" style="display:none">
                <?php $this->_outDialogBox() ?>
            </div>

            <div id="check-holidays" style="display:none">
                <?php $this->_outNationalHolidays($holidays) ?>
            </div>

        </div>
<?php
        ob_end_flush();
    }

    // カレンダー一括操作ボックス
    private function _operation()
    {
        $todayYear = (int) date_i18n('Y');

        // 前年リンク
        $prevLink = $this->oShopCalendar->year - 1;
        if (-4 < ($this->oShopCalendar->year - $todayYear)) {
            $prevLink = sprintf('<a href="%s&amp;y=%d">%d</a>', $this->shopCalendarUrl, $prevLink, $prevLink);
        }

        // 翌年リンク
        $nextLink = $this->oShopCalendar->year + 1;
        if (($this->oShopCalendar->year - $todayYear) < 1) {
            $nextLink = sprintf('<a href="%s&amp;y=%d">%d</a>', $this->shopCalendarUrl, $nextLink, $nextLink);
        }

?>
        <span class="operation-action">
            一括設定：<?php echo $this->oHelper->selectWeek('alter-day-week', '', '', ''); ?>
            <input id="alter-weekly-close" type="button" class="button" value="休業日">
            <input id="alter-weekly-open" type="button" class="button" value="営業日">
        </span>
        <span class="operation-action">
            祝祭日：
            <input id="get-holidays" type="button" class="button" value="読込む">
        </span>

        <span class="operation-action">
            編集年：
            <span class="calendar-link"><?php echo $prevLink ?></span>
            <span class="calendar-link"><?php echo $nextLink ?></span>
        </span>
<?php
    }

    // カレンダー編集表示１２カ月
    private function _calendarAll()
    {
        $oCalendar = new Calendar;
        $oCalendarView = new CalendarView;

        // TDタグに休業日のクラス属性をセットする
        $oCalendarView->callTdClass = array($this, 'setOpenClose');

        // 日付内に営業日データをセットする
        $oCalendarView->callDayContent = array($this, 'setDayData');


        for ($i = 0; $i < 12; $i++) {
            $monthTime = mktime(0, 0, 0, $i + 1, 1, $this->oShopCalendar->year);
            $oCalendar->setCalendar($monthTime);

            echo '<div class="admin-eigyo-calendar">' . "\n";
            echo $oCalendarView->monthCalendar($oCalendar);
            echo "</div>\n";
        }
    }

    /**
     * 休業日のクラス属性をセットする
     */
    public function setOpenClose($datetime, $classes)
    {
        $idx = date('z', $datetime);

        $classes[] = $this->oShopCalendar->shopDay[$idx]['open'] ? '' : 'shop-close';

        if (!empty($this->oShopCalendar->shopDay[$idx]['class'])) {
            $classes[] = $this->oShopCalendar->shopDay[$idx]['class'];
        }

        return $classes;
    }

    /**
     * 営業日データをセットする
     */
    public function setDayData($datetime)
    {
        $data = $this->oShopCalendar->shopDay[date('z', $datetime)];
        $note = isset($data['memo'][Config::LANG]) ? $data['memo'][Config::LANG] : '';

        return sprintf('<div class="day-memo" data-datetime="%d" data-open="%d" data-class="%s" data-memo="%s">%s</div>',
            $datetime, $data['open'], $data['class'], esc_html(json_encode($data['memo'], JSON_FORCE_OBJECT)), $note);
    }

    // 編集ダイアログボックス出力
    private function _outDialogBox()
    {
?>
        <h3 id="shop-day-date"></h3>
        <table class="shop-day-edit">
            <tr>
                <th>店舗</th>
                <td>
                    <?php echo $this->oHelper->radioButton('shop-day-open', '', 'shop_open_close', array('休業', '営業'), 0) ?>
                </td>
            </tr>
            <tr>
                <th>クラス</th>
                <td>
                    <?php echo $this->oHelper->textBox('shop-day-class', '', '', '') ?>
                </td>
            </tr>
            <tr>
                <th>注記</th>
                <td><?php foreach (Config::$languages as $lang => $title) {
                        echo '<div class="line-box">';
                        echo sprintf('<label for="shop-day-memo-%s" class="lang-title">%s</label>', $lang, $title);
                        echo sprintf('<input id="shop-day-memo-%s" class="mts-middle" type="text" data-lang="%s" value="">', $lang, $lang);
                        echo "</div>";
                    } ?></td>
            </tr>
        </table>

<?php
    }

    // 対象年の祝祭日を出力する
    private function _outNationalHolidays($holidays)
    {
        echo sprintf('<input id="national-holidays" type="hidden" value="%s">', empty($holidays) ? '' : esc_html(json_encode($holidays)));

        if (empty($holidays)) :
            echo "祝祭日データを取得できませんでした。";
        else :
?>
        <table id="calendar-holidays">
            <?php foreach ($holidays as $daytime => $title) {
                echo sprintf("<tr><td>%s</td><td>%s</td></tr>\n", date_i18n('Y-n-j', $daytime), $title);
            } ?>
        </table>

<?php
        endif;
    }

    // メッセージ表示
    private function _outMessage()
    {
        static $messageClass = array(
            1 => 'error',
            2 => 'warning',
            3 => 'success',
            4 => 'info',
        );

  ?>
        <div class="notice notice-<?php echo $messageClass[$this->messageNo] ?>">
            <p><?php echo $this->message ?></p>
        </div>

<?php
    }

    /**
     * エラーメッセージを表示する
     */
    public function errorView($errCode, $errSub, $lang='ja')
    {
        $this->setError($errCode, $errSub, $lang);

        ob_start();

        $this->_outMessageBox();

        return ob_get_clean();
    }

    /**
     * エラーメッセージをセットする
     */
    public function setError($errCode, $errSub='')
    {
        $errMessage = array(
                'SAVE_FAILED' => '保存できませんでした。'
        );

        $this->messageNo = 1;
        $this->message = sprintf((isset($errMessage[$errCode]) ? $errMessage[$errCode] : $errCode), $errSub);

        return false;
    }

    /**
     * メッセージをセットする
     */
    public function setMessage($msgCode, $msgSub='')
    {
        static $message = array(
            'SAVED' => '保存しました。',
        );

        $this->messageNo = 3;
        $this->message = sprintf((isset($message[$msgCode]) ? $message[$msgCode] : $msgCode), $msgSub);

        return true;
    }

}
