<?php
/**
 * MTS Car Booking 営業カレンダー
 *
 * @Filename    ShopCalendar.php
 * @Author      S.Hayashi
 * @Code        2018-07-31 Ver.1.0.0
 */
namespace MTSCarBookingTrial\models;

use MTSCarBookingTrial\Config;


class ShopCalendar
{
    private static $shopDay = array(
        //'day' => 0,         // 日付
        'open' => 1,        // 営業日
        'class' => '',      // td属性
        'memo' => array(),
    );

    private $today = null;

    private $data = array(
        'optionName' => '',
        'yearTime' => 0,
        'year' => 0,
        'shopDay' => array(),
    );

    private $memo = array();

    /**
     * Constructor
     */
    public function __construct()
    {
        foreach (Config::$languages as $key => $title) {
            $this->memo[$key] = '';
        }
    }

    /**
     * 営業カレンダー取得の初期化
     */
    public function initShopCalendar($yearTime)
    {
        $ymd = explode('-', date_i18n('Y-n-j', $yearTime));

        $this->data['year'] = (int) $ymd[0];
        $this->data['optionName'] = sprintf('%s_calendar_%04d', Config::DOMAIN, $this->year);
        $this->data['yearTime'] = mktime(0, 0, 0, 1, 1, $this->year);
        $this->data['shopDay'] = array();
    }

    /**
     * 指定年の営業カレンダーを読み込む
     */
    public function readShopCalendar($yearTime)
    {
        $this->initShopCalendar($yearTime);
        $data = get_option($this->optionName);

        if ($data) {
            $this->data['shopDay'] = $data;
            return true;
        }

        return false;
    }

    /**
     * 新しい１カ年データをセットする
     */
    public function newShopCalendar($yearTime)
    {
        // カレンダー以外のプロパティを初期化する
        $this->initShopCalendar($yearTime);

        // 年間日数を計算する
        $days = date('z', mktime(0, 0, 0, 12, 31, $this->year));

        //$shopDay = array();

        // 年通算日でデータを初期設定する
        for ($i = 0; $i <= $days; $i++) {
            $this->data['shopDay'][] = self::$shopDay;
        }
    }

    /**
     * 営業日データをセットする
     */
    public function setShopDay($daytime, $shopDay)
    {
        $days = date('z', $daytime);
        $this->data['shopDay'][$days] = $shopDay;
    }

    /**
     * 指定年の営業カレンダーを保存する
     */
    public function saveShopCalendar()
    {
        if (!add_option($this->optionName, $this->data['shopDay'], '', 'no')) {
            return update_option($this->optionName, $this->data['shopDay']);
        }

        return true;
    }

    /**
     * 指定日付の営業データを取得する
     */
    public function getShopDay($daytime)
    {
        $year = (int) date('Y', $daytime);
        $dayi = (int) date('z', $daytime);

        if ($this->year === $year && isset($this->data['shopDay'][$dayi])) {
            return $this->data['shopDay'][$dayi];
        }

        return false;
    }

    /**
     * 翌年の営業カレンダーを読み込む
     */
    public function readNextYear()
    {
        $oShopCalendar = new ShopCalendar();

        $nextYear = mktime(0, 0, 0, 1, 1, $this->year + 1);

        if ($oShopCalendar->readShopCalendar($nextYear)) {
            return $oShopCalendar;
        }

        return false;
    }

    /**
     * 指定された月の１カ月の営業データを戻す
     */
    public function getShopMonth($monthtime)
    {
        if ($this->year == date('Y', $monthtime)) {
            $start = date('z', $monthtime);

            if (isset($this->data['shopDay'][$start])) {
                return array_slice($this->data['shopDay'], date('z', $monthtime), date('t', $monthtime));
            }
        }

        return array();
    }


    /**
     * プロパティから読み出す
     */
    public function __get($key)
    {
        if (array_key_exists($key, $this->data)) {
            return $this->data[$key];
        }

        if (isset($this->$key)) {
            return $this->$key;
        }

        $trace = debug_backtrace();
        trigger_error(sprintf(
            "Undefined property: '%s&' in %s on line %d, E_USER_NOTICE",
            $key, $trace[0]['file'], $trace[0]['line']
        ));

        return null;
    }

    /**
     * 指定期間の営業カレンダーを取得する
     */
    public static function getShopCalendarDays($daytime, $days)
    {
        $calendar = array();

        $oShopCalendar = new ShopCalendar();
        $endTime = $daytime + 86400 * $days;
        $idxTime = 0;

        for ($shopTime = $daytime; $shopTime < $endTime; $shopTime += 86400) {
            // 年の切り替わりチェック
            if (date('Y', $idxTime) != date('Y', $shopTime)) {
                $oShopCalendar->readShopCalendar($shopTime);
            }
            // 月の切り替わりチェック
            if (date('m', $idxTime) != date('m', $shopTime)) {
                // 月別インデックス
                $idxTime = $shopTime;
            }

            $calendar[$idxTime][$shopTime] = $oShopCalendar->getShopDay($shopTime);
        }

/*
        $oShopCalendar->readShopCalendar($daytime);

        if (!$oShopCalendar->readShopCalendar($daytime)) {
            return false;
        }

        for ($i = 0, $dtime = $daytime; $i < $days; $i++, $dtime += 86400) {
            $shopDay = $oShopCalendar->getShopDay($dtime);

            // 翌年に掛かる場合は翌年のデータを取得する
            if (!$shopDay) {
                $oShopCalendar = $oShopCalendar->readNextYear();
                if (!$oShopCalendar) {
                    break;
                }
                $shopDay = $oShopCalendar->getShopDay($dtime);
            }

            // 指定期間のデータを配列で戻す
            $calendar[$oShopCalendar->year][$dtime] = $shopDay;
        }
*/
        return $calendar;
    }

}
