<?php
/**
 * MTS Car Booking 料金表マスタ(カスタム投稿タイプ)
 *
 * @Filename    Charge.php
 * @Author      S.Hayashi
 * @Code        2018-07-09 Ver.1.0.0
 */
namespace MTSCarBookingTrial\models;

use MTSCarBookingTrial\Config;
use MTSCarBookingTrial\CustomPost;

class Charge
{
    // 種別
    const VEHICLE = 'vehicle';      // 車両レンタル料金
    const OPTION = 'option';        // オプションレンタル料金

    // 計算方法
    const FIXED = 0;                // 定額料金
    const RATE = 1;                 // 乗算
    const TABLE = 2;                // 表引き

    // ポストメタデータカラム
    private static $postmetas = array('name', 'item', 'charge', 'regular', 'limited', 'period');

    // 料金表データ
    private $data = array(
        'chargeId' => 0,            // 車両ポストID
        'name' => array(),          // 表示名称
        'item' => 'vehicle',        // 種別
        'charge' => array(),        // 計算方法、単価・定額料金
        'regular' => array(),       // 料金表
        'limited' => array(),       // 特別料金表
        'period' => array(),        // 特別料金期間
        'oPost' => null,            // カスタムポストデータ
    );

    public $errCode = '';
    public $errSub = '';

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->data['name'] = Config::$languages;

        array_walk($this->data['name'], function(&$val, $key) {
            $val = '';
        });

        // 計算方法、単価・定額料金
        $this->data['charge'] = array(
            'currency' => 'jpy',
            'method' => self::RATE,
            'regular' => array('time' => 86400, 'fee' => 0),
            'limited' => array('time' => 86400, 'fee' => 0),
        );

        // 特別料金期間
        $this->data['period'] = array(
            'start' => 0,
            'end' => 0,
        );
    }

    /**
     * 指定された期間の料金を計算する
     */
    public function charge($start, $end)
    {
        $charge = 0;

        // 特別料金
        if ($this->limitedPeriod($start, $end)) {
            $price = $this->charge->limited['fee'];
            $unitTime = $this->charge->limited['time'];
            $table = $this->limited;
        }

        // 平常料金
        else {
            $price = $this->charge->regular['fee'];
            $unitTime = $this->charge->regular['time'];
            $table = $this->regular;
        }

        // 計算方法別料金計算
        switch ($this->charge->method) {
            case self::FIXED:
                $charge = $price;
                break;
            case self::RATE:
                $charge = ceil(($end - $start) / $unitTime) * $price;
                break;
            case self::TABLE:
                $charge = $this->_tableCharge(($end - $start), $table, $unitTime, $price);
                break;
        }

        return $charge;
    }

    // 表引き料金計算をする
    private function _tableCharge($periodTime, $table, $unitTime, $price)
    {
        $rows = count($table);
        if ($rows <= 0) {
            return $this->_setError('CHARGE_TABLE_UNDEFINED');
        }

        $last = $table[$rows - 1];

        // 料金テーブルを超える利用
        if ($last['time'] < $periodTime) {
            $charge = $last['price']
                + ceil(($periodTime - $last['time']) / $unitTime) * $price;
        }

        // 料金テーブル内の利用
        else {
            for ($i = 0; $i < $rows; $i++) {
                if ($periodTime < $table[$i]['time']) {
                    $charge = $table[$i]['price'];
                    break;
                }
            }
        }

        return $charge;
    }

    // ポストメタデータのキーを戻す
    protected function _metaKey($column)
    {
        return sprintf('%s_%s', CustomPost::CHARGE, $column);
    }

    /**
     * 特別料金期間か確認する
     */
    public function limitedPeriod($startTime, $endTime)
    {
        if (0 < $this->period->start && 0 < $this->period->end) {
            if ($this->period->start <= $startTime && $startTime <= $this->period->end) {
                return true;
            }
        }

        return false;
    }



    /**
     * 料金データを取得する
     */
    public static function readCharge($chargeId)
    {
        $oCharge = new Charge;

        if (is_numeric($chargeId)) {
            $oCharge->oPost = get_post($chargeId);
        } else {
            $oCharge->oPost = $chargeId;
        }

        if ($oCharge->oPost) {
            $oCharge->chargeId = $oCharge->oPost->ID;
            $oCharge->_readPostmeta();
        }

        return $oCharge;
    }

    // ポストメタデータを読み込む
    private function _readPostmeta()
    {
        foreach ($this->data as $column => $data) {
            if (in_array($column, self::$postmetas)) {
                $metaVal = get_post_meta($this->chargeId, $this->_metaKey($column), true);

                if (!empty($metaVal)) {
                    $this->data[$column] = $metaVal;
                }
            }
        }
    }

    /**
     * 料金データを保存する
     */
    public function saveCharge()
    {
        return $this->_savePostmeta();
    }

    // ポストメタデータを保存する
    private function _savePostmeta()
    {
        foreach ($this->data as $column => $data) {
            if (in_array($column, self::$postmetas)) {
                $metaKey = $this->_metaKey($column);

                if (!add_post_meta($this->chargeId, $metaKey, $data, true)) {
                    $oldData = get_post_meta($this->chargeId, $metaKey, true);
                    if ($oldData != $data) {
                        if (!update_post_meta($this->chargeId, $metaKey, $data)) {
                            return $this->_setError('FAILED_UPDATE_POSTMETA', $metaKey);
                        }
                    }
                }
            }
        }

        return true;
    }

    /**
     * 料金データを削除する
     */
    public function deleteCharge()
    {
        return $this->_deletePostmeta();
    }

    // ポストメタデータを削除する
    private function _deletePostmeta()
    {
        foreach ($this->data as $column => $data) {
            if (in_array($column, self::$postmetas)) {
                $metaKey = $this->_metaKey($column);

                if (!delete_post_meta($this->chargeId, $metaKey)) {
                    return $this->_setError('FAILED_DELETE_POSTMETA', $metaKey);
                }
            }
        }

        return true;
    }

    // エラー終了
    private function _setError($errCode, $sub='')
    {
        $this->errCode = $errCode;
        $this->errSub = $sub;

        return false;
    }

    /**
     * データを一括戻す
     */
    public function getData()
    {
        return $this->data;
    }

    /**
     * データを一括セットする
     */
    public function setData($data)
    {
        return $this->data = $data;
    }

    /**
     * プロパティから読み出す
     */
    public function __get($key)
    {
        if (array_key_exists($key, $this->data)) {
            if (in_array($key, array('name', 'charge', 'period'))) {
                return (object) $this->data[$key];
            }
            return $this->data[$key];
        }

        if (isset($this->$key)) {
            return $this->$key;
        }

        $trace = debug_backtrace();
        trigger_error(sprintf(
            "Undefined property: '%s&' in %s on line %d, E_USER_NOTICE",
            $key, $trace[0]['file'], $trace[0]['line']
        ));

        return null;
    }

    /**
     * プロパティをセットする
     */
    public function __set($key, $value)
    {
        if (array_key_exists($key, $this->data)) {
            if (is_array($value)) {
                foreach ($value as $item => $val) {
                    $this->data[$key][$item] = $val;
                }
            } else {
                $this->data[$key] = $value;
            }
        } else {
            $this->$key = $value;
        }

        return $value;
    }

}
