<?php
/**
 * MTS Car Booking 予約データのシンボル化処理
 *
 * @Filename    Alternate.php
 * @Author      S.Hayashi
 * @Code        2019-04-24 Ver.1.0.0
 */
namespace MTSCarBookingTrial\models;

use MTSCarBookingTrial\Config;
use MTSCarBookingTrial\lang\Translation;

class Alternate
{
    private $lang = '';

    public $vars = array();

    /**
     * Constructor
     */
    public function __construct($lang=Config::LANG)
    {
        $this->lang = $lang;
    }

    /**
     * 変数を置換する
     */
    public function replaceVar($str)
    {
        return str_replace(array_keys($this->vars), array_values($this->vars), $str);
    }

    /**
     * 変数を生成する
     */
    public function makeVar(Rental $oRental, Vehicle $oVehicle)
    {
        $payment = $oRental->getPayment('charge');
        $cancel = $oRental->getPayment('cancel');

        $vars = $this->_rentalMailVar($oRental);
        $vars += $this->_paymentVar($payment);
        $vars += $this->_cancelVar($cancel);
        $vars += $this->_vehicleMailVar($oVehicle, $oRental->oVehicleSchedule);
        $vars += $this->_optionMailVar($oRental->optionSchedules, $oVehicle->options);
        $vars += $this->_customerMailVar($oRental->oCustomer);

        $currentTime = current_time('timestamp');
        $vars += array(
            '%CURRENT_DATE%' => $this->_dateStr($currentTime),
            '%CURRENT_TIME%' => $this->_timeStr($currentTime),
        );

        // ユーザー変数追加
        $vars += apply_filters('mtsrcb_add_mail_var', array(), $oRental, $oVehicle);

        $this->vars = $vars;

        return $vars;
    }

    private function _rentalMailVar(Rental $oRental)
    {
        $rentStartDate = $this->_dateStr($oRental->rent_start);
        $rentStartTime = $this->_timeStr($oRental->rent_start);
        $rentEndDate = $this->_dateStr($oRental->rent_end);
        $rentEndTime = $this->_timeStr($oRental->rent_end);

        return array(
            '%RESERVE_ID%' => $oRental->reserve_id,
            '%RENT_START%' => sprintf('%s %s', $rentStartDate, $rentStartTime),
            '%RENT_START_DATE%' => $rentStartDate,
            '%RENT_START_TIME%' => $rentStartTime,
            '%RENT_END%' => sprintf('%s %s', $rentEndDate, $rentEndTime),
            '%RENT_END_DATE%' => $rentEndDate,
            '%RENT_END_TIME%' => $rentEndTime,
            '%NOTE%' => $oRental->note,
        );
    }

    // 支払い
    private function _paymentVar($payment)
    {
        $settingPayment = Config::getSetting('payment');

        $total = isset($payment['total']) ? $payment['total'] : 0;
        $tax = isset($payment['tax']) ? $payment['tax'] : 0;
        $taxRate = isset($payment['taxRate']) ? $payment['taxRate'] : $settingPayment->tax;
        $type = isset($payment['consumption']) ? $payment['consumption'] : $settingPayment->consumption;
        $consumption = $this->_consumption($taxRate, $type);
        $currency = $this->_currencyStr(isset($payment['currency']) ? $payment['currency'] : $settingPayment->currency);
        $grandTotal = isset($payment['grandTotal']) ? $payment['grandTotal'] : 0;

        return array(
            '%PAYMENT_TOTAL%' => $this->_money($total),
            '%PAYMENT_TAX%' => $this->_money($tax),
            '%PAYMENT_TAX_RATE%' => $taxRate,
            '%PAYMENT_CONSUMPTION%' => $consumption,
            '%PAYMENT_CURRENCY%' => $currency,
            '%PAYMENT_GRAND_TOTAL%' => $this->_money($grandTotal),
        );
    }

    private function _currencyStr($currency)
    {
        return Translation::$fCharge[strtoupper($currency)][$this->lang];
    }

    // キャンセル
    private function _cancelVar($cancel)
    {
        $settingPayment = Config::getSetting('payment');

        $total = isset($cancel['total']) ? $cancel['total'] : 0;
        $charge = isset($cancel['cancelCharge']) ? $cancel['cancelCharge'] : 0;
        $tax = isset($cancel['tax']) ? $cancel['tax'] : 0;
        $taxRate = isset($cancel['taxRate']) ? $cancel['taxRate'] : $settingPayment->tax;
        $type = isset($cancel['consumption']) ? $cancel['consumption'] : $settingPayment->consumption;
        $consumption = $this->_consumption($taxRate, $type);
        $cancelTotal = isset($cancel['cancelTotal']) ? $cancel['cancelTotal'] : 0;
        $currency = $this->_currencyStr(isset($cancel['currency']) ? $cancel['currency'] : $settingPayment->currency);

        return array(
            '%CANCEL_TOTAL%' => $total,
            '%CANCEL_CHARGE%' => $charge,
            '%CANCEL_TAX%' => $tax,
            '%CANCEL_TAX_RATE%' => $taxRate,
            '%CANCEL_CONSUMPTION%' => $consumption,
            '%CANCEL_CANCEL_TOTAL%' => $cancelTotal,
            '%CANCEL_CURRENCY' => $currency,
        );
    }

    // 消費税
    private function _consumption($taxRate, $consumption)
    {
        $taxStr = '';

        if ($consumption != 0) {
            $taxStr = sprintf(Translation::$fCharge['RATE'][$this->lang], $taxRate);
            if ($consumption < 0) {
                $taxStr .= Translation::$fCharge['INCLUSIVE'][$this->lang];
            } else {
                $taxStr .= Translation::$fCharge['EXCLUSIVE'][$this->lang];
            }
        }

        return Translation::$fCharge['TAX'][$this->lang] . $taxStr;
    }

    // 車両埋め込み変数
    private function _vehicleMailVar(Vehicle $oVehicle, Schedule $oSchedule)
    {
        $name =  $oVehicle->name->{$this->lang};
        $charge = $this->_money($oSchedule->charge);

        return apply_filters(
            'mtsrcb_mail_vehicle_var',
            array(
                '%VEHICLE_NAME%' => $name,
                '%VEHICLE_NAME_CHARGE%' => sprintf('  %s  %s', $name, $charge),
            ),
            array(
                'name' => $name,
                'charge' => $charge,
            )
        );
    }

    // オプション埋め込み変数
    private function _optionMailVar($optionSchedules, $vehicleOptions)
    {
        $options = $this->_optionVar($optionSchedules, $vehicleOptions);

        $name = $nameCharge = '';

        foreach ($options as $option) {
            $name .= (empty($name) ? '' : "\n") . sprintf("%s", $option['name']);
            $nameCharge .= (empty($nameCharge) ? '' : "\n" ) . sprintf("  %s  %s", $option['name'], $option['charge']);
        }

        return apply_filters(
            'mtsrcb_mail_option_var',
            array(
                '%OPTION_NAME%' => $name,
                '%OPTION_NAME_CHARGE%' => $nameCharge,
            ),
            $options
        );
    }

    private function _optionVar($optionSchedules, $vehicleOptions)
    {
        $options = array();

        foreach ($optionSchedules as $oOptionSchedule) {
            if ($oOptionSchedule->select) {
                $options[] = array(
                    'name' => $vehicleOptions[$oOptionSchedule->option_id]->name->{$this->lang},
                    'charge' => $this->_money($oOptionSchedule->charge),
                );
            }
        }

        return $options;
    }

    // メール埋め込みデータ変数を戻す
    private function _customerMailVar(Customer $oCustomer)
    {

        return array(
            '%COMPANY%' => $oCustomer->company,
            '%SEI%' => $oCustomer->sei,
            '%MEI%' => $oCustomer->mei,
            '%SEI_KANA%' => $oCustomer->sei_kana,
            '%MEI_KANA%' => $oCustomer->mei_kana,
            '%EMAIL%' => $oCustomer->email,
            '%POSTCODE%' => $oCustomer->postcode,
            '%PREF%' => $oCustomer->pref,
            '%CITY%' => $oCustomer->city,
            '%TOWN%' => $oCustomer->town,
            '%BUILDING%' => $oCustomer->building,
            '%COUNTRY%' => $oCustomer->country,
            '%TEL%' => $oCustomer->tel,
            '%MOBILE%' => $oCustomer->mobile,
            '%FAX%' => $oCustomer->fax,
            '%GENDER%' => $oCustomer->gender != '' ? Translation::$customer[$oCustomer->gender][$this->lang] : '',
            '%BIRTHDAY%' => $oCustomer->birthday,
        );
    }

    // 日付
    private function _dateStr($dateTime)
    {
        if ($this->lang == 'ja') {
            return date_i18n(Translation::$dateFormat[$this->lang], $dateTime);
        }

        return date(Translation::$dateFormat[$this->lang], $dateTime);
    }

    // 時刻
    private function _timeStr($dateTime)
    {
        if ($this->lang == 'ja') {
            return date_i18n(Translation::$timeFormat[$this->lang], $dateTime);
        }

        return date(Translation::$timeFormat[$this->lang], $dateTime);
    }

    // 料金
    private function _money($cost)
    {
        $number = explode('.', $cost);

        $amount = number_format($number[0]) . (isset($number[1]) ? sprintf('.%s', $number[1]) : '');

        return $amount;
    }

}
