<?php
/**
 * MTS Car Booking 料金表データ(mtsrcb_charge)管理処理
 *
 * @Filename    ChargeAdmin.php
 * @Author      S.Hayashi
 * @Code        2018-07-07 Ver.1.0.0
 */
namespace MTSCarBookingTrial;

use MTSCarBookingTrial\models\Charge;
use MTSCarBookingTrial\views\ChargeAdminView;
use MTSCarBookingTrial\views\ParamAdmin;

class ChargeAdmin
{
    const PAGE_NAME = CarBooking::ADMIN_CHARGE;

    // View
    private $view = null;

    /**
     * Constructor
     */
    public function __construct()
    {
        // 管理画面 カスタム投稿タイプ一覧処のカラム処理追加
        add_filter('manage_' . CustomPost::CHARGE . '_posts_columns', array($this, 'postsColumns'));
        add_action('manage_' . CustomPost::CHARGE . '_posts_custom_column', array($this, 'postsCustomColumn'), 10, 2);

        // 料金表ビューの編集フック登録
        add_action ("add_meta_boxes_" . CustomPost::CHARGE, array($this, 'loadView'));

        // 料金表の入力と保存
        add_action('save_post', array($this, 'savePost'), 10, 3);

        // 料金表の削除
        add_action('before_delete_post', array($this, 'deletePost'));
    }

    /**
     * 管理画面 料金データ一覧のカラムタイトル
     */
    public function postsColumns($columns)
    {
        $planColumns =  array(
            'item' => '対象',
        );

        return array_merge(array_slice($columns, 0, 2), $planColumns, array_slice($columns, 2));
    }

    /**
     * 管理画面 料金データ一覧のカラムデータ
     */
    public function postsCustomColumn($column, $chargeId)
    {

        $oCharge = Charge::readCharge($chargeId);

        if ($oCharge) {
            if ($column === 'item') {
                if (array_key_exists($oCharge->item, ParamAdmin::$chargeItem)) {
                    echo ParamAdmin::$chargeItem[$oCharge->item];
                } else {
                    echo 'ー';
                }
            }
        }
    }

    /**
     * 編集処理準備
     */
    public function loadView($oPost)
    {
        $oCharge = Charge::readCharge($oPost);

        $this->view = new ChargeAdminView($oCharge);
    }

    /**
     * 料金表データ(postmeta)の入力保存処理
     */
    public function savePost($postId, $post, $update)
    {
        // カスタム投稿タイプ、操作権限を確認する
        if (get_post_type($postId) != CustomPost::CHARGE || !current_user_can(CustomPost::CAPABILITY, $postId)) {
            return;
        }

        // nonceチェック
        if (!isset($_POST['charge_nonce']) || !wp_verify_nonce($_POST['charge_nonce'], self::PAGE_NAME)) {
            return;
        }

        // Auto saveの場合は保存しないようにする
        //if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        //    return;
        //}

        $oCharge = Charge::readCharge($post);

        // 編集データの入力
        $this->_input($oCharge);

        $result = $oCharge->saveCharge();

        if (!$result) {
            die(sprintf('Error: %s on %s in %s.', $oCharge->errCode, $oCharge->errSub, get_class($this)));
        }

    }

    // 料金表データの読み込み
    private function _input(Charge $oCharge)
    {
        $post = stripslashes_deep($_POST['charge']);

        $data = $oCharge->getData();

        foreach ($data as $column => &$val) {
            switch ($column) {
                case 'name':
                case 'item':
                    $val = $post[$column];
                    break;
                case 'charge':
                    $val['method'] = intval($post['charge']['method']);
                    $val['currency'] = array_key_exists($post['charge']['currency'], ParamAdmin::$currency)
                        ? $post['charge']['currency'] : $val['currency'];
                    $val['regular'] = $this->_inputPrice($post['charge']['regular']);
                    $val['limited'] = $this->_inputPrice($post['charge']['limited']);
                    break;
                case 'regular':
                case 'limited':
                    $val = json_decode($post[$column], true);
                    break;
                case 'period':
                    $val['start'] = $this->_inputDate($post['period']['start']);
                    $val['end'] = $this->_inputDate($post['period']['end']);
                    break;
            }
        }

        $oCharge->setData($data);
    }

    // 単価料金の読み込み
    private function _inputPrice($post)
    {
        $time = intval($post['time']) * ($post['unit'] == 'day' ? 86400 : 3600);
        $fee = intval($post['fee']);

        return compact('time', 'fee');
    }

    // 期日の読み込み
    private function _inputDate($post)
    {
        if (0 < $post['year'] && 0 < $post['month'] && 0 < $post['day']) {
            return strtotime(sprintf('%d-%d-%d', $post['year'], $post['month'], $post['day']));
        }

        return 0;
    }

    /**
     * 料金表データ(postmeta)の削除
     */
    public function deletePost($postId)
    {
        // カスタム投稿タイプ、操作権限を確認する
        if (get_post_type($postId) != CustomPost::CHARGE || !current_user_can(CustomPost::CAPABILITY, $postId)) {
            return;
        }

        $oCharge = Charge::readCharge($postId);

        $result = $oCharge->deleteCharge();
/*
        if (!$result) {
            die(sprintf('Error: %s on %s in %s.', $oCharge->errCode, $oCharge->errSub, get_class($this)));
        }
*/
    }

    /**
     * 料金表のタイトル一覧を戻す
     */
    public static function listCharge($item)
    {
        global $wpdb;

        $sql = $wpdb->prepare("SELECT ID,post_title "
        . "FROM {$wpdb->posts} "
        . "LEFT JOIN {$wpdb->postmeta} ON ID=post_id "
        . "WHERE post_status=%s AND post_type=%s AND {$wpdb->postmeta}.meta_value=%s "
        . "ORDER BY menu_order",
            'publish', CustomPost::CHARGE, $item);

        $list = array();

        $data = $wpdb->get_results($sql);

        if ($data) {
            foreach ($data as $charge) {
                $list[$charge->ID] = $charge->post_title;
            }
        }

        return $list;
    }

}
