<?php
/**
 * MTS Car Booking レンタル予定管理
 *
 * @Filename    BudgetAdmin.php
 * @Author      S.Hayashi
 * @Code        2018-08-16 Ver.1.0.0
 */
namespace MTSCarBookingTrial;

use MTSCarBookingTrial\models\Calendar;
use MTSCarBookingTrial\models\ShopCalendar;
use MTSCarBookingTrial\models\Vehicle;
use MTSCarBookingTrial\models\Schedule;
use MTSCarBookingTrial\views\BudgetAdminView;

class BudgetAdmin
{
    const PAGE_NAME = CarBooking::ADMIN_BUDGET;

    const DAYS = 15;
    const DELTA = 7;

    // View
    private $view = NULL;

    // カレンダー操作オブジェクト
    public $oCalendar = null;

    // 営業カレンダー
    public $shopDays = array();

    // 車両スケジュール
    public $schedules = array();

    // 予約入力
    public $oSchedule = null;

    public $msgCode = '';
    public $msgSub = '';
    public $errflg = false;

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->oCalendar = new Calendar;
        $this->oSchedule = new Schedule;
        $this->view = new BudgetAdminView;
    }

    /**
     * レンタル予定管理
     */
    public function budget()
    {
        // カレンダー表示先頭日付 Unix Time
        $dtime = isset($_GET['pt']) ? intval($_GET['pt']) : $this->oCalendar->todayTime;

        $ymd = explode('-', date('Y-n-j', $dtime));
        $startTime = mktime(0, 0, 0, $ymd[1], $ymd[2], $ymd[0]);

        if ($ymd[0] < Config::RELEASE_YEAR) {
            $startTime = $this->oCalendar->todayTime;
        }

        $endTime = mktime(0, 0, 0, $ymd[1], $ymd[2] + self::DAYS, $ymd[0]);

        // レンタル予約データを取得する
        $this->_getRentalParam();

        // カレンダー操作オブジェクト設定
        $this->oCalendar->setCalendar($startTime, self::DELTA);

        // 指定期間の営業カレンダーデータ
        $this->shopDays = ShopCalendar::getShopCalendarDays($startTime, self::DAYS);

        // 期間中のレンタル車両を検索する
        $this->schedules = Schedule::findRentalVehicle($startTime, $endTime);

        $this->view->budgetView($this);
    }

    // レンタル予約データを取得する
    private function _getRentalParam()
    {
        if (isset($_GET['vehicle_id'])) {
            $this->oSchedule->vehicle_id = (int) $_GET['vehicle_id'];
            if (isset($_GET['rent_start'])) {
                $this->oSchedule->rent_start = (int) $_GET['rent_start'];
            }
            if (isset($_GET['rent_end'])) {
                $this->oSchedule->rent_end = (int) $_GET['rent_end'];
            }
        }
    }

    /**
     * 営業カレンダー表示
     */
    public function shopCalendar()
    {
        $oShopCalendar = new ShopCalendar;

        $thisYear = (int)date_i18n('Y');
        $year = isset($_GET['y']) ? intval($_GET['y']) : $thisYear;

        if (isset($_POST['save_calendar']) && wp_verify_nonce($_POST['calendar_admin_nonce'], self::PAGE_NAME)) {
            if ($this->_inputCalendar($oShopCalendar)) {
                if ($oShopCalendar->saveShopCalendar()) {
                    $this->view->setMessage('SAVED');
                } else {
                    $this->view->setError('SAVE_FAILED');
                }
            }
            $year = $oShopCalendar->year;
            $yearTime = $oShopCalendar->yearTime;
        } else {
            if (($year - $thisYear) < -4 || ($thisYear + 1) < $year) {
                $year = $thisYear;
            }

            $yearTime = mktime(0, 0, 0, 1, 1, $year);

            if (!$oShopCalendar->readShopCalendar($yearTime)) {
                $oShopCalendar->newShopCalendar($yearTime);
            }
        }

        $endTime = mktime(0, 0, 0, 1, 1, $year + 1);
        $holidays = self::getHolidays($yearTime, $endTime);

        $this->view->editView($oShopCalendar, $holidays);
    }

    // 送信された編集カレンダーを取り込む
    private function _inputCalendar(ShopCalendar $oShopCalendar)
    {
        // 送信データを取得する
        $shopCalendar = json_decode(stripslashes($_POST['shop_calendar']), true);

        // オブジェクトを初期化する
        $oShopCalendar->initShopCalendar(key($shopCalendar));

        // 営業データをセットする
        foreach ($shopCalendar as $daytime => $shopDay) {
            $oShopCalendar->setShopDay($daytime, $shopDay);
        }

        return true;
    }




    /**
     * Googleカレンダーから祝祭日を取得する
     */
    public static function getHolidays($startTime, $endTime)
    {
        $cafile = dirname(dirname(__DIR__)) . DIRECTORY_SEPARATOR . 'ca-bundle.crt';

        $options = array(
            'ssl' => array(
                'cafile' => $cafile,
                'verify_peer' => false,
                'verify_peer_name' => false,
            ),
        );

        $iCal = @file_get_contents(self::$iCalFile, false, stream_context_create($options));

        if (empty($iCal)) {
            return false;
        }

        $iCal = preg_replace("/\r\n|\r|\n/", "\n", $iCal);

        // 祝日一覧配列を取得する
        $holidays = self::_getHolidays($iCal, $startTime, $endTime);

        return $holidays;
    }

    // 祝日一覧配列を取得する
    private static function _getHolidays($iCal, $startTime, $endTime)
    {
        // 祝日一覧配列を生成する
        $pos = 0;
        $dateTime = '';
        $holidays = array();

        while ($str = self::_getLine($pos, $iCal)) {
            if (empty($dateTime)) {
                if (preg_match("/^DTSTART;VALUE=DATE:(\d*)$/", $str, $matches)) {
                    $unixTime = strtotime(sprintf('%s-%s-%s', substr($matches[1], 0, 4), substr($matches[1], 4, 2), substr($matches[1], 6, 2)));
                    if ($startTime <= $unixTime && $unixTime < $endTime) {
                        $dateTime = $unixTime;
                    }
                }
            } elseif (preg_match("/^SUMMARY:(.*)$/", $str, $matches)) {
                $holidays[$dateTime] = $matches[1];
                $dateTime = '';
            }

            $pos += mb_strlen($str) + 1;
        }

        // 日付順にソートする
        ksort($holidays, SORT_NUMERIC);

        return $holidays;
    }

    // 1行取得する
    private static function _getLine($pos, $string)
    {
        // 次の文字位置
        $nextPos = mb_strpos($string, "\n", $pos);

        if ($nextPos !== false) {
            return mb_substr($string, $pos, $nextPos - $pos);
        }

        return false;
    }

}
