<?php
/**
 * MTS Car Booking レンタル表示のビュー
 *
 * @Filename    RentalCarView.php
 * @Author      S.Hayashi
 * @Code        2018-12-19 Ver.1.0.0
 */
namespace MTSCarBookingTrial\views;

use MTSCarBookingTrial\Config;
use MTSCarBookingTrial\CarBooking;
use MTSCarBookingTrial\lang\Translation;
use MTSCarBookingTrial\models\Rental;
use MTSCarBookingTrial\models\Vehicle;
use MTSCarBookingTrial\models\Customer;

class RentalCarView
{
    protected $lang = Config::LANG;

    /**
     * メッセージを表示する
     */
    protected function _dispMessage($msgCode, $msgSub, $errflg)
    {
        // メッセージ文をセットする
        if (isset(Translation::$fMessage[$msgCode][$this->lang])) {
            $message = sprintf(Translation::$fMessage[$msgCode][$this->lang], $msgSub);
        } else {
            $message = $msgCode . ($msgSub ? " : {$msgSub}" : '');
        }

        echo sprintf('<div id="mts-message-pane" class="%s">%s</div>',
                ($errflg ? 'error' : 'update'), esc_html($message)) . "\n";
    }

    /**
     * 予約メインデータを表示する
     */
    protected function _dispRentalPeriod(Rental $oRental, Vehicle $oVehicle, $title='FORM_BOOKING_TITLE')
    {

?>
        <div class="rental-form-block">
            <?php if (!empty(Translation::$bForm[$title][$this->lang])) {
                echo sprintf('<h3 class="rental-form-title">%s</h3>', Translation::$bForm[$title][$this->lang]); } ?>
            <table id="mts-rental-application" class="mts-form-table">
                <?php if ($oRental->reserve_id != '') : ?><tr class="rental-reserve-id">
                    <th><div class="reserve-id-title"><?php echo Translation::$rForm['RESERVE_ID'][$this->lang] ?></div></th>
                    <td><div class="reserve-id-number"><?php echo $oRental->reserve_id ?></div></td>
                </tr><?php endif; ?>
                <tr class="rental-reserve-period">
                    <th><div class="reserve-period-title"><?php echo Translation::$bForm['FORM_PERIOD_TITLE'][$this->lang] ?></div></th>
                    <td>
                        <div class="reserve-period-datetime">
                            <div class="rental-period-date"><?php $this->_dispDateTime($oRental->rent_start) ?></div>
                            <div class="rental-period-arrow"><?php echo Translation::$pArrow[$this->lang] ?></div>
                            <div class="rental-period-date"><?php $this->_dispDateTime($oRental->rent_end) ?></div>
                        </div>
                    </td>
                </tr>
                <tr class="rental-reserve-vehicle">
                    <th><div class="reserve-vehicle-title"><?php echo Translation::$bForm['FORM_VEHICLE_TITLE'][$this->lang] ?></div></th>
                    <td>
                        <div class="reserve-vehicle-name"><?php echo $oVehicle->name->{$this->lang} ?></div>
                        <?php $imgNo = apply_filters('mtsrcb_booking_form_img', 0, $title);
                            if (isset($oVehicle->image[$imgNo]) && $oVehicle->image[$imgNo] != '') {
                                echo sprintf('<div class="reserve-vehicle-image"><img class="rental-vehicle-img" src="%s">', $oVehicle->image[$imgNo]);
                    } ?></td>
                </tr>
            </table>
        </div>

<?php
    }

    // 予約日時の表示
    private function _dispDateTime($datetime)
    {
        if ($this->lang == 'ja') {
            $date = date_i18n(Translation::$dateFormat[$this->lang], $datetime);
        } else {
            $date = date(Translation::$dateFormat[$this->lang], $datetime);
        }

        $time = date(Translation::$timeFormat[$this->lang], $datetime);

        echo sprintf('<span class="rental-period-day">%s</span>', $date);
        echo sprintf('<span class="rental-period-time">%s</span>', $time);
    }

    /**
     * 規約ページ・プライバシーポリシーページのリンク表示
     */
    protected function _dispGdprLink()
    {
        $settingForm = Config::getSetting('form');

        $link = '';

        if ($settingForm->tos_url != '') {
            $link .= sprintf('<div id="link-tos_url" class="form-page-link">%s<a href="%s" target="_blank">%s</a></div>',
                Translation::$lArrow[$this->lang], $settingForm->tos_url, Translation::$bForm['FORM_TERMS_OF_USE'][$this->lang]) . "\n";
        }

        if ($settingForm->gdpr_url != '') {
            $link .= sprintf('<div id="link-gdpr_url" class="form-page-link">%s<a href="%s" target="_blank">%s</a></div>',
                Translation::$lArrow[$this->lang], $settingForm->gdpr_url, Translation::$bForm['FORM_PRIVACY_POLICY'][$this->lang]) . "\n";
        }

        if (empty($link)) {
            return;
        }
?>
        <div class="rental-form-block gdpr-link">
            <?php echo $link ?>
        </div>
<?php
    }

    /**
     * オプションを表示する
     */
    protected function _dispOptionList($optionSchedules, $options)
    {
        if (empty($options)) {
            return;
        }

        $selected = 0;
?>
        <div class="rental-form-block option-list">
            <h3 class="rental-form-title"><?php echo Translation::$bForm['FORM_OPTION_TITLE'][$this->lang] ?></h3>

            <table id="mts-option-confirm" class="mts-form-table">
                <?php foreach ($optionSchedules as $oOptionSchedule) {
                    if ($oOptionSchedule->select) {
                        echo sprintf('<tr><th>%s</th>', $options[$oOptionSchedule->option_id]->name->{$this->lang});
                        echo sprintf('<td class="option-selected">%s</td></tr>', Translation::$bForm['FORM_OPTION_RENTAL'][$this->lang]);
                        $selected++;
                    }
                }

                if ($selected <= 0) {
                    echo sprintf('<tr><td class="option-selected"><span class="not-selected">%s</span></td></tr>',
                        Translation::$bForm['FORM_NOT_SELECTED'][$this->lang]);
                } ?>

            </table>
        </div>

<?php
    }

    /**
     * 料金計算結果を表示する
     */
    protected function _dispCharge(Rental $oRental, Vehicle $oVehicle)
    {
        $settingForm = Config::getSetting('form');

        // 料金表示しない
        if ($settingForm->charge <= 0) {
            return;
        }

        // 料金の合計表を取得する
        $totalList = (object) $oRental->getPayment('charge');

        // 料金設定の通貨種類により小数点以下桁数を決める
        $decimals = isset(Config::$currency[$totalList->currency]) ? Config::$currency[$totalList->currency] : 0;

?>
        <div class="rental-form-block charge-list">
            <h3><?php echo Translation::$fCharge['CHARGE_TITLE'][$this->lang] ?></h3>
            <table class="mts-form-table">
                <tr>
                    <th class="rental-product vehicle"><?php echo $oVehicle->name->{$this->lang} ?></th>
                    <td class="rental-charge vehicle"><?php echo number_format($oRental->oVehicleSchedule->charge, $decimals) ?></td>
                </tr>
                <?php foreach($oRental->optionSchedules as $oSchedule) :
                    if ($oSchedule->select) : ?><tr>
                    <th class="rental-product option"><?php echo $oVehicle->options[$oSchedule->option_id]->name->{$this->lang} ?></th>
                    <td class="rental-charge option"><?php echo number_format($oSchedule->charge, $decimals) ?></td>
                </tr><?php endif; endforeach; ?>
                <?php if ($totalList->consumption != 0) : ?><tr>
                    <th class="rental-product total"><?php echo Translation::$fCharge['SUBTOTAL'][$this->lang] ?></th>
                    <td class="rental-charge total"><?php echo number_format($totalList->total, $decimals) ?></td>
                </tr><?php endif; ?>
                <?php if ($totalList->consumption != 0) : ?><tr>
                    <th class="rental-product tax"><?php echo sprintf(Translation::$fCharge['TAX'][$this->lang],
                        $totalList->taxRate, Translation::$fCharge[0 < $totalList->consumption ? 'EXCLUSIVE' : 'INCLUSIVE'][$this->lang]) ?></th>
                    <td class="rental-charge tax"><?php echo number_format($totalList->tax, $decimals) ?></td>
                </tr><?php endif; ?>
                <tr>
                    <th class="rental-product grand-total"><?php echo Translation::$fCharge['TOTAL'][$this->lang] ?></th>
                    <td class="rental-charge grand-total"><?php
                        echo number_format($totalList->grandTotal, $decimals)?></td>
                </tr>
            </table>
            <div class="rental-currency"><?php echo sprintf('%s : <span class="currency-type">%s</span>',
                Translation::$fCharge['CURRENCY'][$this->lang], Translation::$fCharge[strtoupper($totalList->currency)][$this->lang]) ?></div>
        </div>

<?php
    }

    /**
     * 画像アップロードフォーム出力
     */
    protected function _dispUpload(Rental $oRental)
    {
        $settingForm = Config::getSetting('form');

        if ($settingForm->passport <= 0 && $settingForm->license <= 0) {
            return;
        };

        ?>
        <div class="rental-form-block">
            <h3><?php echo Translation::$bForm['FORM_UPLOAD_TITLE'][$this->lang] ?></h3>
            <table class="mts-form-table">
                <?php
                $this->_outImageInput('passport', $settingForm->passport, $oRental->upload['passport']);
                $this->_outImageInput('license', $settingForm->license, $oRental->upload['license']);
                ?>
            </table>
        </div>

        <?php

    }

    // 画像アップロード入力フォーム
    private function _outImageInput($item, $disp, $graphicsId)
    {
        // 画像アップロード不要
        if ($disp <= 0) {
            return;
        }

        $title = sprintf('FORM_%s', strtoupper($item));
        $required = 1 < $disp ? Translation::$others['required'][$this->lang] : '';
        ?>
        <tr>
            <th><?php echo sprintf('<label for="rental-%s">%s%s</label>', $item, Translation::$bForm[$title][$this->lang], $required) ?></th>
            <td><?php if (0 < $graphicsId) {
                    echo sprintf('<div class="image-uploaded">%s<input type="hidden" value="%d"></div>', Translation::$bForm['FORM_IMAGE_UPLOADED'][$this->lang], $graphicsId);
                } else {
                    echo sprintf('<input id="rental-%s-%s" type="file" accept="image/*" name="rental_%s">', $item, $this->lang, $item);
                }?></td>
        </tr>

        <?php
    }



    /**
     * 画像アップロード状態の出力
     */
    protected function _dispUploadState(Rental $oRental)
    {
        $settingForm = Config::getSetting('form');

        if ($settingForm->passport <= 0 && $settingForm->license <= 0) {
            return;
        };

?>
        <div class="rental-form-block upload-state">
            <h3><?php echo Translation::$bForm['FORM_UPLOAD_TITLE'][$this->lang] ?></h3>
            <table class="mts-form-table">
                <?php
                    $this->_outImageState('passport', $settingForm->passport, $oRental->upload['passport']);
                    $this->_outImageState('license', $settingForm->license, $oRental->upload['license']);
                ?>
            </table>
        </div>

<?php
    }

    // 画像アップロード状態
    private function _outImageState($item, $disp, $graphicsId)
    {
        // 画像アップロード不要
        if ($disp <= 0) {
            return;
        }

        $title = sprintf('FORM_%s', strtoupper($item));
        $required = 1 < $disp ? Translation::$others['required'][$this->lang] : '';
        $registerd = 0 < $graphicsId ? 'uploaded' : 'registerd';
?>
        <tr>
            <th><?php echo sprintf('%s%s', Translation::$bForm[$title][$this->lang], $required) ?></th>
            <td><?php echo sprintf('<span class="image-%s">%s</span>',
                $registerd, Translation::$bForm['FORM_IMAGE_' . strtoupper($registerd)][$this->lang]); ?>
            </td>
        </tr>

<?php
    }

    /**
     * 予約申込者のビュー
     */
    protected function _customerView(Customer $oCustomer, $view='form')
    {
        $oCustomerView = new CustomerView;

        $setting = Config::getSetting('customer');
        //$oCustomer->apiKey = $setting->api_key;

?>
        <div class="rental-form-block customer-<?php echo $view ?>">
            <h3 class="rental-form-title"><?php echo Translation::$bForm['FORM_APPLICANT_TITLE'][$this->lang] ?></h3>
            <?php if ($view == 'form') {
                if (!empty(Translation::$bForm['FORM_REQUIRED_DESCRIPTION'][$this->lang])) {
                    echo sprintf('<div class="form-required description">%s</div>', Translation::$bForm['FORM_REQUIRED_DESCRIPTION'][$this->lang]);
                }
                $oCustomerView->formView($oCustomer, (array) $setting);
            } else {
                $oCustomerView->confirmationView($oCustomer, (array) $setting);
            } ?>
        </div>

<?php
    }

    /**
     * 決済処理と決済処理リンクの表示
     */
    protected function _dispPayment(Rental $oRental)
    {
        // 決済情報を取得する
        $payment = $oRental->paymentAvailable();

        // 決済機能を利用しない　または　決済受付期間外
        if ($payment === false) {
            return;
        }

        // 支払い状況
        $state = 0 < $oRental->accounting ? 'PAID' : 'UNPAID';

?>
        <div class="rental-form-block charge-payment">
            <h3><?php echo Translation::$fCharge['PAYMENT_TITLE'][$this->lang] ?></h3>
            <table class="mts-form-table">
                <tr>
                    <th class="payment-item payment-state"><?php echo Translation::$fCharge['PAYMENT_STATE'][$this->lang] ?></th>
                    <td class="payment-data payment-state"><?php echo Translation::$fCharge['PAYMENT_' . $state][$this->lang] ?></td>
                </tr>
                <tr>
                    <th class="payment-item payment-period"><?php echo Translation::$fCharge['PAYMENT_PERIOD'][$this->lang] ?></th>
                    <td class="payment-data payment-period"><?php
                        echo sprintf('<span class="period-date">%s</span>', $this->_formatDate($payment['assigned']));
                        echo sprintf('<span class="period-arrow">%s</span>', Translation::$pArrow[$this->lang]);
                        echo sprintf('<span class="period-date">%s</span>', $this->_formatDate($payment['limit'])); ?></td>
                </tr>
            </table>
        </div>
        <?php if ($state === 'UNPAID') : ?><div class="rental-form-block action-row">
            <input type="submit" class="button-primary" name="reserve_payment" value="<?php echo Translation::$fButton['PAY'][$this->lang] ?>">
        </div>

<?php
        endif;
    }

    /**
     * 言語に合わせた日付表示を戻す
     */
    protected function _formatDate($time)
    {
        if ($this->lang === 'ja') {
            return date_i18n(Translation::$dateFormat[$this->lang], $time);
        }

        return date(Translation::$dateFormat[$this->lang], $time);
    }

}