<?php
/**
 * MTS Car Booking 車両マスタ(カスタム投稿タイプ)
 *
 * @Filename    Vehicle.php
 * @Author      S.Hayashi
 * @Code        2018-07-03 Ver.1.0.0
 */
namespace MTSCarBookingTrial\models;

use MTSCarBookingTrial\CustomPost;

class Vehicle
{
    const NUMBER_OF_IMAGES = 2;

    // 名称
    private static $carName = array(
        'ja' => '',                 // 日本語
        'en' => '',                 // 英語
    );

    // ナンバー
    private static $carNumber = array(
        'region' => '',             // 地域名
        'class' => '',              // 分類番号
        'kana' => '',               // 平仮名
        'designation' => '',        // 一連指定番号
    );

    // 料金情報
    private static $chargeDetail = array(
        'charge_id' => 0,           // 料金表ポスト
        'fuel' => 'regular',        // 油種
        'fuel_km' => 0,             // 燃費
    );

    // ポストメタデータカラム
    private static $postmetas = array('name', 'number', 'expiration', 'charge', 'detail', 'image');

    // 車両データ
    private $data = array(
        'postId' => 0,           // 車両ポストID
        'name' => array(),          // 表示名称
        'number' => array(),        // ナンバー
        'expiration' => '',         // 有効期限満了日
        'charge' => array(),        // 料金情報
        'detail' => array(),        // 詳細情報
        'image' => array(),         // 車両画像
        'oPost' => null,            // カスタムポストデータ
    );

    // 車両種別配列(id=>name)
    public $terms = array();

    // 選択対象オプションオブジェクト一覧(車種が同一 = term_relationshipsのterm_taxonomy_idが等しい)
    public $options = array();

    // 料金データ
    public $oCharge = null;

    public $errCode = '';
    public $errSub = '';

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->data['name'] = self::$carName;
        $this->data['number'] = self::$carNumber;
        //$this->data['detail'] = self::$carDetail;
        $this->data['charge'] = self::$chargeDetail;
        for ($i = 0; $i < self::NUMBER_OF_IMAGES; $i++) {
            $this->data['image'][$i] = '';
        }
    }

    // ポストメタデータのキーを戻す
    protected function _metaKey($column)
    {
        return sprintf('%s_%s', CustomPost::VEHICLE, $column);
    }

    /**
     * 車両データを取得する
     */
    public static function readVehicle($postId)
    {
        $oVehicle = new Vehicle;

        if (is_numeric($postId)) {
            $oPost = get_post($postId);
        } else {
            $oPost = $postId;
        }

        // postデータを確認する
        if (empty($oPost) || $oPost->post_type != CustomPost::VEHICLE) {
            return $oVehicle;
        }

        $oVehicle->oPost = $oPost;

        if ($oVehicle->oPost) {
            $oVehicle->postId = $oVehicle->oPost->ID;
            $oVehicle->_readPostmeta();
        }

        // 車両種別IDを取得する
        $oVehicle->_getTerms();

        // オプションデータの読み込み
        $oVehicle->_readOptions();

        // 料金データの読み込み
        if (0 < $oVehicle->charge->charge_id) {
            $oVehicle->oCharge = Charge::readCharge($oVehicle->charge->charge_id);
        }

        return $oVehicle;
    }

    // 車両種別IDを取得する
    private function _getTerms()
    {
        global $wpdb;

        $sql = $wpdb->prepare(
            "SELECT term_id FROM {$wpdb->term_taxonomy} WHERE term_taxonomy_id IN " .
                "(SELECT term_taxonomy_id FROM {$wpdb->term_relationships} " .
                "WHERE object_id=%d)", $this->postId);

        $data = $wpdb->get_results($sql, ARRAY_A);

        if ($data) {
            foreach ($data as $termTaxonomy) {
                $this->terms[$termTaxonomy['term_id']] = get_term($termTaxonomy['term_id']);
            }
        }

        return $this->terms;
    }

    // 車種カテゴリが同じオプションデータの一覧を取得する
    private function _readOptions()
    {
        global $wpdb;

        // term_relationshipsからオプションID一覧を取得する
        $sql = $wpdb->prepare(
            "SELECT ID FROM {$wpdb->posts} WHERE ID IN " .
                "(SELECT object_id FROM {$wpdb->term_relationships} " .
                "WHERE object_id<>%d AND term_taxonomy_id IN " .
                    "(SELECT term_taxonomy_id FROM {$wpdb->term_relationships} " .
                    "WHERE object_id=%d)) " .
                "ORDER BY menu_order ASC", $this->postId, $this->postId);

        $data = $wpdb->get_results($sql, ARRAY_A);

        if ($data) {
            foreach ($data as $relationship) {
                $this->options[$relationship['ID']] = Option::readOption($relationship['ID']);
            }
        }

        $this->options;
    }

    // ポストメタデータを読み込む
    private function _readPostmeta()
    {
        foreach ($this->data as $column => $data) {
            if (in_array($column, self::$postmetas)) {
                $metaVal = get_post_meta($this->postId, $this->_metaKey($column), true);

                if (!empty($metaVal)) {
                    $this->data[$column] = $metaVal;
                }
            }
        }
    }

    /**
     * ポストメタデータを保存する
     */
    public function savePostmeta()
    {
        foreach ($this->data as $column => $data) {
            if (!in_array($column, self::$postmetas)) {
                continue;
            }

            $metaKey = $this->_metaKey($column);

            if (!add_post_meta($this->postId, $metaKey, $data, true)) {
                $oldData = get_post_meta($this->postId, $metaKey, true);
                if ($oldData != $data) {
                    if (!update_post_meta($this->postId, $metaKey, $data)) {
                        return $this->_setError('FAILED_UPDATE_POSTMETA', $metaKey);
                    }
                }
            }
        }

        return true;
    }

    /**
     * ポストメタデータを削除する
     */
    public function deletePostmeta()
    {
        foreach ($this->data as $column => $data) {
            if (in_array($column, self::$postmetas)) {
                $metaKey = $this->_metaKey($column);

                if (!delete_post_meta($this->postId, $metaKey)) {
                    return $this->_setError('FAILED_DELETE_POSTMETA', $metaKey);
                }
            }
        }

        return true;
    }

    // エラー終了
    private function _setError($errCode, $sub)
    {
        $this->errCode = $errCode;
        $this->errSub = $sub;

        return false;
    }

    /**
     * データを戻す
     */
    public function getData()
    {
        return $this->data;
    }

    /**
     * データを一括セットする
     */
    public function setData($data)
    {
        return $this->data = $data;
    }

    /**
     * プロパティから読み出す
     */
    public function __get($key)
    {
        if (array_key_exists($key, $this->data)) {
            if (is_array($this->data[$key]) && $key !== 'image') {
                return (object) $this->data[$key];
            }
            return $this->data[$key];
        }

        if (isset($this->$key)) {
            return $this->$key;
        }

        $trace = debug_backtrace();
        trigger_error(sprintf(
            "Undefined property: '%s&' in %s on line %d, E_USER_NOTICE",
            $key, $trace[0]['file'], $trace[0]['line']
        ));

        return null;
    }

    /**
     * プロパティをセットする
     */
    public function __set($key, $value)
    {
        if (array_key_exists($key, $this->data)) {
            if (is_array($value)) {
                foreach ($value as $item => $val) {
                    $this->data[$key][$item] = $val;
                }
            } else {
                $this->data[$key] = $value;
            }
        } else {
            $this->$key = $value;
        }

        return $value;
    }

    /**
     * 車両名一覧を取得する
     */
    public static function listVehicle()
    {
        global $wpdb;

        $order = 'ASC';

        $sql = $wpdb->prepare(
            "SELECT ID,post_title "
            . "FROM {$wpdb->posts} "
            . "WHERE post_type=%s AND post_status=%s "
            . "ORDER BY menu_order {$order}", CustomPost::VEHICLE, 'publish');

        $data = $wpdb->get_results($sql, ARRAY_A);

        return self::_dataList($data);
    }

    // 読み込みデータを一覧リストにして戻す
    private static function _dataList($data)
    {
        $vehicles = array();

        if ($data) {
            foreach ($data as $vehicle) {
                $vehicles[$vehicle['ID']] = $vehicle['post_title'];
            }
        }

        return $vehicles;
    }

    /**
     * 特定車種の車両一覧を取得する
     */
    public static function listVehicleByType($term_id)
    {
        global $wpdb;

        $order = 'ASC';

        $sql = $wpdb->prepare(
            "SELECT ID,post_title FROM {$wpdb->posts} WHERE ID IN "
            . "(SELECT object_id FROM {$wpdb->term_relationships} WHERE term_taxonomy_id IN "
            . "(SELECT term_taxonomy_id FROM {$wpdb->term_taxonomy} WHERE term_id=%d)) "
            . "AND post_type=%s AND post_status=%s "
            . "ORDER BY menu_order {$order}", $term_id, CustomPost::VEHICLE, 'publish');

        $data = $wpdb->get_results($sql, ARRAY_A);

        return self::_dataList($data);
    }

    /**
     * 車種一覧を取得する
     *
     * return array(array(term_id => name), ...)
     */
    public static function listVehicleType()
    {
        global $wpdb;

        $sql = $wpdb->prepare(
            "SELECT ter.term_id AS term_id,ter.name AS name "
            . "FROM $wpdb->terms AS ter"
            . "JOIN $wpdb->term_taxonomy AS tax ON ter.term_id=tax.term_id "
            . "WHERE tax.taxonomy=%s "
            . "ORDER BY ter.term_id ASC", CustomPost::VHEICLE_TYPE);

        $data = $wpdb->get_results($sql, ARRAY_A);

        $list = array();

        if ($data) {
            foreach ($data as $term) {
                $list[$term['term_id']] = $term['name'];
            }
        }

        return $list;
    }

}
