<?php
/**
 * MTS Car Booking レンタル台帳
 *
 * @Filename    Rental.php
 * @Author      S.Hayashi
 * @Code        2018-08-21 Ver.1.0.0
 */
namespace MTSCarBookingTrial\models;

use MTSCarBookingTrial\Config;


class Rental
{
    const MODULE = 'system';
    const MODEL = 'rental';
    const VERSION = '1.0';

    // 予約処理状態
    const CANCELED = -1;            // キャンセル済み
    const BOOKED = 0;               // 予約済み

    // 画像データリンク
    private static $upload = array(
        'passport' => 0,
        'license' => 0,
    );

    private $data = array(
        'rental_id' => 0,
        'status' => 0,              // 予約処理状態
        'accounting' => 0,          // 清算状況
        'reserve_id' => '',         // ユーザー定義(32byte)
        'user_id' => 0,
        'rent_start' => 0,          // 貸渡日日時
        'rent_end' => 0,            // 返却予定日時
        'charge_total' => 0,        // 合計金額
        'payment' => array(),       // 決済データ
        'customer' => array(),      // 予約申込者
        'upload' => array(),        // 画像データ
        'note' => '',               // 注記
        'lang' => 'ja',             // 言語
        'acceptance' => 0,          // 規約同意
        'updated' => '',            // 更新日
        'created' => '',            // 予約申込日
    );

    private $tableName = '';

    // 予約申込者データ
    public $oCustomer = null;

    // 予約スケジュールデータ
    public $schedules = array();

    // スケジュールデータ
    public $oVehicleSchedule = null;    // 車両
    public $optionSchedules = array();  // オプション

    // 画像データ
    //public $image = array();            // Graphicsオブジェクト配列

    public $errCode = '';
    public $errSub = '';

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->tableName = Config::getTableName(self::MODEL);

        $this->upload = self::$upload;

        $this->oCustomer = new Customer;
    }

    /**
     * 車両、選択オプションのレンタルチェック
     */
    public function rentalAvailable(Vehicle $oVehicle)
    {
        $available = true;

        // 車両の予約チェック
        if (!$this->oVehicleSchedule->vehicleAvailable()) {
            $available = false;
        }

        // オプションの予約チェック
        foreach ($this->optionSchedules as $oOptionSchedule) {
            if (!$oOptionSchedule->rentalAvailable($oVehicle->options[$oOptionSchedule->option_id])) {
                $available = false;
            }
        }

        return $available;
    }

    /**
     * レンタルデータオブジェクトを戻す
     */
    public static function readRental($rentalId)
    {
        $oRental = new Rental;

        $oRental->getRental($rentalId);

        return $oRental;
    }

    /**
     * レンタルデータを読込む
     */
    public function getRental($rentalId)
    {
        global $wpdb;

        $tableName = Config::getTableName(self::MODEL);
        $setting = Config::getSetting('customer');

        $data = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM {$tableName} WHERE rental_id=%d", $rentalId));

        if ($data) {
            $this->rental_id = (int) $data->rental_id;
            $this->status = (int) $data->status;
            $this->accounting = (int) $data->accounting;
            $this->reserve_id = $data->reserve_id;
            $this->user_id = (int) $data->user_id;
            $this->rent_start = (int) $data->rent_start;
            $this->rent_end = (int) $data->rent_end;
            $this->charge_total = $data->charge_total;
            $this->payment = unserialize($data->payment);
            $this->customer = $data->customer;
            $this->upload = unserialize($data->upload);
            $this->note = $data->note;
            $this->lang = $data->lang;
            $this->acceptance = $data->acceptance;
            $this->updated = $data->updated;
            $this->created = $data->created;

            $this->oCustomer->setDecryptData($data->customer, $setting->key);
        } else {
            return $this->setError('FAILED_SELECT_RENTAL', $rentalId);
        }

        // 登録スケジュールデータを取得する
        $this->schedules = Schedule::findRental($rentalId);

        // 車両・オプションIDをセットする
        $oVehicleSchedule = null;

        foreach ($this->schedules as $scheduleId => $oSchedule) {
            if (0 < $oSchedule->vehicle_id) {
                $oVehicleSchedule = $oSchedule;
            } elseif (0 < $oSchedule->option_id) {
                $oSchedule->select = true;
                $this->optionSchedules[$scheduleId] = $oSchedule;
            }
        }

        // 車両スケジュールがない場合は空のスケジュールデータをセットする
        $this->oVehicleSchedule = empty($oVehicleSchedule) ? $this->newSchedule() : $oVehicleSchedule;

        return true;
    }

    /**
     * レンタルデータを新規登録する
     */
    public function newRental()
    {
        // レンタルデータ新規追加
        if (!$this->_addRental()) {
            return false;
        }

        // 車両スケジュール新規追加
        $this->oVehicleSchedule->rental_id = $this->rental_id;
        if (!$this->oVehicleSchedule->addSchedule()) {
            return $this->setError($this->oVehicleSchedule->errCode, $this->oVehicleSchedule->vehile_id);
        }

        // オプションスケジュール新規追加
        foreach ($this->optionSchedules as $oOptionSchedule) {
            if ($oOptionSchedule->select) {
                $oOptionSchedule->rental_id = $this->rental_id;
                if (!$oOptionSchedule->addSchedule()) {
                    return $this->setError($oOptionSchedule->errCode, $oOptionSchedule->option_id);
                }
            }
        }

        // アップロードしたパスポート画像データにレンタルIDをセットする
        if (0 < $this->upload['passport']) {
            if (!Graphics::updateRentalId($this->upload['passport'], $this->rental_id)) {
                return $this->setError('UPDATE_GRAPHICS_RENTAL_ID', $this->upload['passport']);
            }
        }

        // アップロードしたライセンス画像データにレンタルIDをセットする
        if (0 < $this->upload['license']) {
            if (!Graphics::updateRentalId($this->upload['license'], $this->rental_id)) {
                return $this->setError('UPDATE_GRAPHICS_RENTAL_ID', $this->upload['license']);
            }
        }

        return true;
    }

    /**
     * 予約日付を合わせた新しいスケジュールデータを用意する
     */
    public function newSchedule($vehicleId=0, $optionId=0)
    {
        $oSchedule = new Schedule;

        $oSchedule->vehicle_id = $vehicleId;
        $oSchedule->option_id = $optionId;

        $this->setSchedule($oSchedule);

        return $oSchedule;
    }

    /**
     * スケジュールデータにレンタル情報をセットする
     */
    public function setSchedule(Schedule &$oSchedule)
    {
        $oSchedule->rental_id = $this->rental_id;
        $oSchedule->user_id = $this->user_id;
        $oSchedule->rent_start = $this->rent_start;
        $oSchedule->rent_end = $this->rent_end;

        return;
    }

    // レンタルデータの新規追加
    private function _addRental()
    {
        global $wpdb;

        // 予約フォーム申込者設定
        $settingCustomer = Config::getSetting('customer');

        // 新規登録日時
        $this->created = $this->updated = current_time('mysql');

        $result = $wpdb->insert(
            $this->tableName,
            array(
                'status' => $this->status,
                'accounting' => $this->accounting,
                'reserve_id' => $this->reserve_id,
                'user_id' => $this->user_id,
                'rent_start' => $this->rent_start,
                'rent_end' => $this->rent_end,
                'charge_total' => $this->charge_total,
                'payment' => serialize($this->payment),
                'customer' => $this->oCustomer->getEncryptData($settingCustomer->key),
                'upload' => serialize($this->upload),
                'note' => $this->note,
                'lang' => $this->lang,
                'acceptance' => $this->acceptance,
                'updated' => $this->updated,
                'created' => $this->created,
            ),
            array(
                '%d', '%d', '%s', '%d', '%d', '%d', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%s', '%s'
            )
        );

        if (!$result) {
            return $this->setError('FAILED_INSERT_RENTAL');
        }

        $this->rental_id = $wpdb->insert_id;

        // 予約IDを新しく登録する
        if ($this->reserve_id != '' || $this->_newReserveId()) {
            $this->clearError();
            return $this->rental_id;
        }

        return false;

    }

    // 新しい予約IDを登録する
    private function _newReserveId()
    {
        global $wpdb;

        $pow = intval(Config::RESERVE_ID_RANDOM);
        $rand = 0 < $pow ? rand(pow(10, $pow), (pow(10, $pow + 1) - 1)) : Config::RESERVE_ID_RANDOM;

        $reserveId = sprintf('%s%s%s',
            $rand,
            date_i18n('ymd', strtotime($this->created)),
            substr('000' . $this->rental_id, -4));

        // 置換フィルター
        $this->reserve_id = apply_filters('mtsrcb_reserve_id', $reserveId, $this);

        $this->updated = current_time('mysql');

        $res = $wpdb->update(
            $this->tableName,
            array(
                'reserve_id' => $this->reserve_id,
                'updated' => $this->updated,
            ),
            array('rental_id' => $this->rental_id),
            array('%s', '%s'),
            array('%d')
        );

        if (!$res) {
            $this->setError('FAILED_UPDATE_RESERVE_ID', $this->rental_id);
        }

        $this->clearError();
        return $res;
    }

    /**
     * レンタルデータを更新する
     */
    public function updateRental()
    {
        // レンタルデータの更新
        if (!$this->_saveRental()) {
            return false;
        }

        // 車両が選択されていない場合はスキップする
        if (0 < $this->oVehicleSchedule->vehicle_id) {
            // 車両スケジュールデータの更新
            if (0 < $this->oVehicleSchedule->schedule_id) {
                if (!$this->oVehicleSchedule->saveSchedule()) {
                    return $this->setError($this->oVehicleSchedule->errCode, $this->oVehicleSchedule->errSub);
                }
            }
            // 車両スケジュールデータの新規登録
            else {
                if (!$this->oVehicleSchedule->addSchedule()) {
                    return $this->setError($this->oVehicleSchedule->errCode, $this->oVehicleSchedule->errSub);
                }
            }
        }

        // オプションスケジュールの新規追加・更新
        foreach ($this->optionSchedules as $oOptionSchedule) {
            // 非選択オプションの確認
            if (!$oOptionSchedule->select) {
                continue;
            }

            // オプションスケジュールの新規登録
            if ($oOptionSchedule->schedule_id <= 0) {
                //$oOptionSchedule->rental_id = $this->rental_id;
                if (!$oOptionSchedule->addSchedule()) {
                    return $this->setError($oOptionSchedule->errCode, $oOptionSchedule->errSub);
                }

            // オプションスケジュールデータの更新
            } else {
                if (!$oOptionSchedule->saveSchedule()) {
                    return $this->setError($oOptionSchedule->errCode, $oOptionSchedule->errSub);
                }
            }
        }

        // キャンセルされたオプションを削除する
        foreach ($this->optionSchedules as &$oOptionSchedule) {
            // 選択オプションの確認
            if ($oOptionSchedule->select) {
                continue;
            } else {
                if (0 < $oOptionSchedule->schedule_id ) {
                    if ($oOptionSchedule->deleteSchedule()) {
                        // 新規スケジュールデータへ差し替える
                        $oOptionSchedule = $this->newSchedule(0, $oOptionSchedule->option_id);
                    } else {
                        return $this->setError($oOptionSchedule->errCode, $oOptionSchedule->errSub);
                    }
                }
            }
        }

        // 更新終了
        return true;
    }

    // レンタルデータを保存する
    private function _saveRental()
    {
        global $wpdb;

        // 予約フォーム申込者設定
        $settingCustomer = Config::getSetting('customer');

        // 編集登録日時
        $this->updated = current_time('mysql');

        $res = $wpdb->update(
            $this->tableName,
            array(
                'status' => $this->status,
                'accounting' => $this->accounting,
                'reserve_id' => $this->reserve_id,
                'user_id' => $this->user_id,
                'rent_start' => $this->rent_start,
                'rent_end' => $this->rent_end,
                'charge_total' => $this->charge_total,
                'payment' => serialize($this->payment),
                'customer' => $this->oCustomer->getEncryptData($settingCustomer->key),
                'upload' => serialize($this->upload),
                'note' => $this->note,
                'lang' => $this->lang,
                'acceptance' => $this->acceptance,
                'updated' => $this->updated,
            ),
            array('rental_id' => $this->rental_id),
            array(
                '%d', '%d', '%s', '%d', '%d', '%d', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%s'
            ),
            array('%d')
        );

        if (!$res) {
            $this->setError('FAILED_UPDATE_RENTAL', $this->rental_id);
        }

        $this->clearError();
        return $res;
    }

    /**
     * レンタルデータを削除する
     */
    public function deleteRental()
    {
        global $wpdb;

        // アップロード画像を削除する
        foreach ($this->upload as $type => $graphicsId) {
            if (!$this->deleteGraphics($type, $this->upload[$type])) {
                return false;
            }
        }

        // スケジュールデータを削除する
        if (!$this->_deleteSchedules()) {
            return false;
        }

        // レンタルデータを削除する
        $res = $wpdb->delete(
            $this->tableName,
            array('rental_id' => $this->rental_id),
            array('%d')
        );

        if (!$res){
            return $this->_setError('FAILED_DELETE_RENTAL', $this->rental_id);
        }

        return true;
    }

    /**
     * アップロード画像データを削除する
     */
    public function deleteGraphics($type, $graphicsId)
    {
        $oGraphics = new Graphics;

        $oGraphics->graphics_id = $graphicsId;

        // 画像データを削除する
        if (!$oGraphics->deleteGraphics()) {
            return $this->setError($oGraphics->errCode, $oGraphics->errSub);
        }

        $this->data['upload'][$type] = 0;

        return true;
    }

    // スケジュールデータを一括削除する
    private function _deleteSchedules()
    {
        foreach ($this->schedules as $oSchedule) {
            if (!$oSchedule->deleteSchedule()) {
                return $this->setError($oSchedule->errCode, $oSchedule->errSub);
            }
        }

        return true;
    }

    /**
     * レンタル予約を中止する
     */
    public function abortRental()
    {
        $ret = true;

        // 新規予約でなければ画像削除はしない
        if ($this->rental_id <= 0) {
            // 画像データを削除する
            foreach ($this->data['upload'] as $key => $gid) {
                if (0 < $gid) {
                    $ret = $this->deleteGraphics($key, $gid);
                }
            }
        }

        return $ret;
    }

    /**
     * 予約データのキャンセル
     */
    public function cancelRental()
    {
        // レンタルデータにキャンセルをセットする
        $this->status = self::CANCELED;

        // キャンセルデータの更新処理をする
        if (!$this->updateCancel()) {
            return false;
        }

        // スケジュールデータを削除する
        if (!$this->_deleteSchedules()) {
            return false;
        }

        return true;
    }

    /**
     * キャンセルデータの更新処理をする
     */
    public function updateCancel()
    {
        global $wpdb;

        $this->updated = current_time('mysql');

        $res = $wpdb->update(
            $this->tableName,
            array(
                'status' => $this->status,
                'payment' => serialize($this->payment),
                'updated' => $this->updated,
            ),
            array('rental_id' => $this->rental_id),
            array('%d', '%s', '%s'),
            array('%d')
        );

        if (!$res) {
            $this->setError('FAILED_UPDATE_CANCEL', $this->rental_id);
        }

        return $res;
    }

    // エラー終了
    public function setError($errCode, $errSub='')
    {
        $this->errCode = $errCode;
        $this->errSub = $errSub;
        return false;
    }

    // エラーをクリアする
    public function clearError()
    {
        $this->errCode = $this->errSub ='';
    }

    /**
     * レタンル予約データの清算状態を変更する
     */
    public function updateAccounting($accounting)
    {
        global $wpdb;

        $updated = current_time('mysql');

        $res = $wpdb->update(
            $this->tableName,
            array(
                'accounting' => $accounting,
                'updated' => $updated,
            ),
            array('rental_id' => $this->rental_id),
            array('%d', '%s'),
            array('%d')
        );

        if ($res) {
            $this->accounting = $accounting;
            $this->updated = $updated;
        } else {
            $this->setError('FAILED_UPDATE_ACCOUNTING', $this->rental_id);
        }

        return $res;
    }

    /**
     * 支払いデータを更新する
     */
    public function updatePayment()
    {
        global $wpdb;

        $this->updated = current_time('mysql');

        $res = $wpdb->update(
            $this->tableName,
            array(
                'charge_total' => $this->charge_total,
                'payment' => serialize($this->payment),
                'updated' => $this->updated,
            ),
            array('rental_id' => $this->rental_id),
            array('%s', '%s', '%s'),
            array('%d')
        );

        if (!$res) {
            $this->setError('FAILED_UPDATE_PAYMENT', $this->rental_id);
        }

        return $res;
    }

    /**
     * 決済データに新しい明細データを上書きする
     */
    public function newPayment()
    {
        $payment = array();

        // 車両
        $payment[$this->oVehicleSchedule->schedule_id] = $this->oVehicleSchedule->charge;

        // オプション
        foreach ($this->optionSchedules as $oOptionSchedule) {
            if ($oOptionSchedule->select) {
                $payment[$oOptionSchedule->schedule_id] = $oOptionSchedule->charge;
            }
        }

        $totalList = $this->calculateTotal();

        $payment['total'] = $totalList->total;
        $payment['tax_rate'] = $totalList->taxRate;
        $payment['tax'] = $totalList->tax;
        //$payment['fuel'] = 0;

        return $this->updatePayment($payment, $totalList->chargeTotal);
    }


    /**
     * 支払いデータを更新する
     */
    public function updateUpload()
    {
        global $wpdb;

        $updated = current_time('mysql');

        $res = $wpdb->update(
            $this->tableName,
            array(
                'upload' => serialize($this->upload),
                'updated' => $updated,
            ),
            array('rental_id' => $this->rental_id),
            array('%s', '%s'),
            array('%d')
        );

        if ($res) {
            //$this->upload = $upload;
            $this->updated = $updated;
        } else {
            $this->setError('FAILED_UPDATE_UPLOAD', $this->rental_id);
        }

        return $res;
    }

    /**
     * 画像アップロードファイルをデータベースに保存、レンタルデータに登録する
     */
    public function saveGraphics($type, $files)
    {
        $oGraphics = new Graphics;

        $ret = true;

        // 登録済み画像データを検索する
        $oGraphics->findGraphics($this->rental_id, $type);

        // アップロード画像ファイルを読み込む
        if (!$oGraphics->retrieveFile($type, $files)) {
            return $this->setError($oGraphics->errCode, $oGraphics->errSub);
        }

        // レンタルIDをセットする
        $oGraphics->rental_id = $this->rental_id;

        // 登録済みデータなら画像データを更新する
        if (0 < $oGraphics->graphics_id) {
            if (!$oGraphics->updateGraphics()) {
                return $this->setError($oGraphics->errCode, $oGraphics->errSub);
            }
        }

        // 未登録なら画像データを新規登録する
        else {
            if (!$oGraphics->addGraphics()) {
                return $this->setError($oGraphics->errCode, $type);
            }
        }

        // 画像データのIDをセット
        $this->data['upload'][$type] = $oGraphics->graphics_id;

        // 予約登録済みなら続けて更新する
        if (0 < $this->rental_id) {
            $ret = $this->updateUpload();
        }

        return $ret;
    }

    /**
     * 予約IDからレンタルデータを検索する
     */
    public static function findByReserveId($reserveId, $check=true)
    {
        global $wpdb;

        $tableName = Config::getTableName(self::MODEL);

        $sql = $wpdb->prepare(
            "SELECT rental_id,status,rent_start FROM {$tableName} WHERE reserve_id like %s", $reserveId);

        $data = $wpdb->get_results($sql, ARRAY_A);

        if (empty($data)) {
            return 0;
        }

        // 予約状態、検索日による出力を抑制する
        if ($check) {
            $currentTime = current_time('timestamp');
            if ($data[0]['status'] < 0 || $data[0]['rent_start'] < $currentTime) {
                return 0;
            }
        }

        return intval($data[0]['rental_id']);
    }

    // 検索条件
    private static function _whereSql($find)
    {
        $where = '';

        if (0 < $find['vehicle_id']) {
            $where = sprintf(" WHERE schedule.vehicle_id=%d", $find['vehicle_id']);
        } elseif (0 < $find['rental_id']) {
            $where = sprintf(" WHERE rental.rental_id=%d", $find['rental_id']);
        }

        if (0 < $find['rent_start']) {
            $rentDate = sprintf(' (rental.rent_start>=%d AND rental.rent_start<%d) OR (rental.rent_end>=%d AND rental.rent_end<%d)',
                $find['rent_start'], $find['rent_end'], $find['rent_start'], $find['rent_end']);

            $where .= (empty($where) ? ' WHERE ' : ' AND ') . $rentDate;
        }

        return $where;
    }

    // ソート
    private static function _orderSql($order)
    {
        // ソート
        if (isset($order['rent_start'])) {
            $orderBy = sprintf('rental.rent_start %s', $order['rent_start']);
        } elseif (isset($order['rental.rent_end'])) {
            $orderBy = sprintf('rental.rent_end %s', $order['rent_end']);
        } elseif (isset($order['rental.rental_id'])) {
            $orderBy = sprintf('rental.rental_id %s', $order['rental_id']);
        } else {
            $orderBy = 'rental_id DESC';
        }

        return $orderBy;
    }

    /**
     * レンタルデータのレコード数カウント
     */
    public static function countRental($condition=array())
    {
        global $wpdb;

        $tableName = Config::getTableName(self::MODEL);
        $scheduleTable = Config::getTableName(Schedule::MODEL);

        $where = self::_whereSql($condition['find']);

        if (0 < $condition['find']['vehicle_id']) {
            $sql = sprintf(
                "SELECT count(*) FROM %s AS rental "
                . "LEFT JOIN %s AS schedule ON rental.rental_id=schedule.rental_id"
                . "{$where}", $tableName, $scheduleTable);
        } else {
            $sql = sprintf("SELECT count(*) FROM %s AS rental", $tableName);
        }

        $data = $wpdb->get_col($sql);

        return intval($data);
    }

    /**
     * レンタルデータの読み込み
     */
    public static function listRental($condition, $limit, $offset)
    {
        global $wpdb;

        $tableName = Config::getTableName(self::MODEL);
        $scheduleTable = Config::getTableName(Schedule::MODEL);

        $where = self::_whereSql($condition['find']);
        $orderBy = self::_orderSql($condition['order']);

        if (0 < $condition['find']['vehicle_id']) {
            $sql = $wpdb->prepare(
                "SELECT rental.rental_id AS rental_id FROM {$tableName} AS rental "
                . "LEFT JOIN {$scheduleTable} AS schedule ON rental.rental_id=schedule.rental_id"
                . "{$where} "
                . "ORDER BY {$orderBy} LIMIT %d OFFSET %d", $limit, $offset);
        } else {
            $sql = $wpdb->prepare(
                "SELECT rental.rental_id AS rental_id FROM {$tableName} AS rental "
                . "{$where} "
                . "ORDER BY {$orderBy} LIMIT %d OFFSET %d", $limit, $offset);
        }

        $data = $wpdb->get_results($sql, ARRAY_A);

        return $data;
    }

    /**
     * セッションデータを保存する
     */
    public function saveSession(Session $oSession)
    {
        $oSession->save(self::MODEL, array(
            'data' => $this->getData(),
            'property' => array(
                'schedule_vehicle' => $this->_getScheduleVehicle(),
                'schedule_options' => $this->_getScheduleOption(),
                'schedules' => $this->_getScheduleData(),
            ),
        ));
    }

    public function _getScheduleVehicle()
    {
        return array(
            'data' => $this->oVehicleSchedule->getData(),
            'select' => $this->oVehicleSchedule->select,
        );
    }


    private function _getScheduleOption()
    {
        $options = array();

        foreach ($this->optionSchedules as $oOptionSchedule) {
            $options[] = array(
                'data' => $oOptionSchedule->getData(),
                'select' => $oOptionSchedule->select,
            );
        }

        return $options;
    }

    private function _getScheduleData()
    {
        $schedules = array();

        foreach ($this->schedules as $oSchedule) {
            $schedules[] = $oSchedule->getData();
        }

        return $schedules;
    }

    /**
     * セッションデータを戻す
     */
    public function restoreSession(Session $oSession)
    {
        $data = $oSession->read(self::MODEL);

        if (!$data) {
            return;
        }

        // レンタルデータ
        $this->setData($data['data']);

        // 車両スケジュール
        $this->oVehicleSchedule = new Schedule;
        $this->oVehicleSchedule->setData($data['property']['schedule_vehicle']['data']);
        $this->oVehicleSchedule->select = $data['property']['schedule_vehicle']['select'];

        // オプションスケジュール
        $optionSchedules = array();
        foreach ($data['property']['schedule_options'] as $scheduleOption) {
            $oSchedule = new Schedule;
            $oSchedule->setData($scheduleOption['data']);
            $oSchedule->select = $scheduleOption['select'];
            $optionSchedules[] = $oSchedule;
        }
        $this->optionSchedules = $optionSchedules;

        // 保存スケジュールデータ
        foreach ($data['property']['schedules'] as $scheduleData) {
            $oSchedule = new Schedule;
            $oSchedule->setData($scheduleData);
            $this->schedules[$oSchedule->schedule_id] = $oSchedule;
        }
    }

    /**
     * データを配列で一括取得する
     */
    public function getData($encrypt=true)
    {
        // 予約フォーム申込者設定
        $settingCustomer = Config::getSetting('customer');

        $data = $this->data;
        if ($encrypt) {
            $data['customer'] = $this->oCustomer->getEncryptData($settingCustomer->key);
        }
        return $data;
    }

    /**
     * データを一括セットする
     */
    public function setData($data, $decrypt=true)
    {
        // 予約フォーム申込者設定
        $settingCustomer = Config::getSetting('customer');

        if ($decrypt) {
            $this->oCustomer->setDecryptData($data['customer'], $settingCustomer->key);
        }

        $this->data = $data;
    }

    /**
     * Paymentデータを取得する
     */
    public function getPayment($key)
    {
        return isset($this->data['payment'][$key]) ? $this->data['payment'][$key] : '';
    }

    /**
     * metaデータをセットする
     */
    public function setPayment($key, $val)
    {
        $this->data['payment'][$key] = $val;
    }

    /**
     * プロパティから読み出す
     */
    public function __get($key)
    {
        if (array_key_exists($key, $this->data)) {
            return $this->data[$key];
        }

        if (isset($this->$key)) {
            return $this->$key;
        }

        $trace = debug_backtrace();
        trigger_error(sprintf(
            "Undefined property: '%s&' in %s on line %d, E_USER_NOTICE",
            $key, $trace[0]['file'], $trace[0]['line']
        ));

        return null;
    }

    /**
     * プロパティをセットする
     */
    public function __set($key, $value)
    {
        if (array_key_exists($key, $this->data)) {
            $this->data[$key] = $value;
        } else {
            $this->$key = $value;
        }

        return $value;
    }

    /**
     * レンタルテーブルの生成とアップデート
     */
    public static function installTable(&$version, $list)
    {
        // レンタルテーブル名を取得する
        $tableName = Config::getTableName(self::MODEL);

        // テーブルが登録されていない、またはバージョンが異なれば更新する
        if (array_key_exists(self::MODEL, $version) && in_array($tableName, $list)) {
            if ($version[self::MODEL] === self::VERSION) {
                return false;
            }
        }

        $sql = "CREATE TABLE $tableName (
            rental_id int(11) unsigned NOT NULL AUTO_INCREMENT,
            status smallint(6) DEFAULT '0',
            accounting smallint(6) DEFAULT '0',
            reserve_id varchar(32) DEFAULT '',
            user_id int(11) DEFAULT '0',
            rent_start bigint(12) DEFAULT '0',
            rent_end bigint(12) DEFAULT '0',
            charge_total varchar(12) DEFAULT '',
            payment text,
            customer text,
            upload text,
            note text,
            lang varchar(32) DEFAULT '',
            acceptance smallint(6) DEFAULT '0',
            updated datetime DEFAULT NULL,
            created datetime DEFAULT NULL,
            PRIMARY KEY  (rental_id),
            KEY reserve_id (reserve_id),
            KEY user_id (user_id),
            KEY rent_start (rent_start),
            KEY rent_end (rent_end)
        ) DEFAULT CHARSET=utf8mb4;";

        dbDelta($sql);

        // バージョンデータの更新
        $version[self::MODEL] = self::VERSION;

        return true;
    }

    /**
     * レンタル料金を計算、セットする
     */
    public function charging(Vehicle $oVehicle)
    {
        // 車両のレンタル料金
        $this->oVehicleSchedule->charge = $oVehicle->oCharge->charge($this->rent_start, $this->rent_end);

        // オプションのレンタル料金
        foreach ($this->optionSchedules as $oOptionSchedule) {
            if ($oOptionSchedule->select) {
                // 料金計算の可否確認と料金データの設定確認
                if (0 < $oVehicle->options[$oOptionSchedule->option_id]->calculation
                 && 0 < $oVehicle->options[$oOptionSchedule->option_id]->charge_id) {
                    $oOptionSchedule->charge = $oVehicle->options[$oOptionSchedule->option_id]->oCharge->charge($this->rent_start, $this->rent_end);
                } else {
                    $oOptionSchedule->charge = 0;
                }
            }
        }

        // 合計計算と計算値をpaymentにセットする
        $this->calculateCharge();
    }

    /**
     * レンタル料金を計算する
     */
    public function calculateCharge()
    {
        $settingPayment = Config::getSetting('payment');

        // 車両の料金
        $total = $this->oVehicleSchedule->charge;

        // オプションのレンタル料金
        foreach ($this->optionSchedules as $oOptionSchedule) {
            if ($oOptionSchedule->select) {
                $total += $oOptionSchedule->charge;
            }
        }

        // 消費税
        $tax = $this->computeTax($total);

        // 総合計
        $grandTotal = 0 <= $settingPayment->consumption ? $total + $tax : $total;

        $charge = array(
            'total' => $total,
            'taxRate' => $settingPayment->tax,
            'tax' => $tax,
            'consumption' => $settingPayment->consumption,
            'grandTotal' => $grandTotal,
            'currency' => $settingPayment->currency,
        );

        // 支払い明細を取得、paymentにセットする
        $this->setPayment('charge', $charge);

        // 請求合計額をセットする
        $this->charge_total = $charge['grandTotal'];
    }

    /**
     * 消費税額計算
     */
    public function computeTax($total)
    {
        // 料金設定
        $settingPayment = Config::getSetting('payment');

        // 通貨の小数点以下桁数
        $decimals = Config::$currency[$settingPayment->currency];

        // 消費税
        $tax = 0;

        if ($settingPayment->consumption != 0) {
            $taxRate = $settingPayment->tax / 100;

            // 内税額
            if ($settingPayment->consumption < 0) {
                $tax = $total - $total / (1 + $taxRate);
            }
            // 外税額
            elseif (0 < $settingPayment->consumption) {
                $tax = $total * $taxRate;
            }

            // 小数点以下を通貨に合わせて切り捨てる
            $cent = 0 < $decimals ? pow(10, $decimals) : 0;
            $tax = $decimals ? floor($tax * $cent) / $cent : floor($tax);
        }

        return $tax;
    }

    /**
     * キャンセル可能か確認する
     */
    public function cancelAvailable($cancelTime=0)
    {
        $settingCancel = Config::getSetting('cancel');

        // キャンセル不可
        if ($settingCancel->limit <= 0) {
            return false;
        }

        // 貸渡日
        $rentalTime = $this->rent_start - $this->rent_start % 86400;

        // キャンセル有効期限日時
        $limitTime = $rentalTime - ($settingCancel->limit - 1) * 86400 - 1;

        // 現在日時
        if ($cancelTime <= 0) {
            $cancelTime = current_time('timestamp');
        }

        // キャンセル有効期限切れ
        if ($limitTime < $cancelTime) {
            return false;
        }

        // キャンセル期限年月日、残り日数
        return array(
            'cancelTime' => $cancelTime,
            'limit' => $limitTime,
            'remain' => ceil(($limitTime - $cancelTime) / 86400),
        );
    }

    /**
     * キャンセル料データを作成する
     */
    public function cancelCharge($cancelTime=0)
    {
        $settingForm = Config::getSetting('form');
        $settingPayment = Config::getSetting('payment');
        $settingCancel = Config::getSetting('cancel');

        // キャンセル実行時刻
        if ($cancelTime === 0) {
            $cancelTime = current_time('timestamp');
        }

        // レンタル日までの日数
        $cancelAvailable = $this->cancelAvailable($cancelTime);
        if (!$cancelAvailable) {
            return $cancelAvailable;
        }

        // キャンセル料表から対応日付を求める
        foreach (Config::$cancelCharge as $days => $cost) {
            if ($cancelAvailable['remain'] <= $days) {
                break;
            }
        }

        // 料金データ
        $total = $cancelCharge = $tax = 0;

        if ($settingForm->charge) {

            $charge = (object)$this->getPayment('charge');

            // レンタル料の計算対象金額は、消費税計算する場合は税引額、しない場合は総合計
            if (0 < $settingCancel->tax) {
                // 内税の場合は合計額から内税額を減額する
                $total = 0 < $settingPayment->consumption ? $charge->total : $charge->total - $charge->tax;
            } else {
                $total = $charge->grandTotal;
            }

            // キャンセル料金を求める
            $cancelCharge = 0;

            if (0 < $total && isset(Config::$cancelCharge[$days])) {
                // 100分率計算
                if ($settingCancel->charge == 1) {
                    $cent = 0 < Config::$currency[$charge->currency] ? pow(10, Config::$currency[$charge->currency]) : 0;
                    $cancelCharge = $cent <= 0 ? floor($total * Config::$cancelCharge[$days]) : floor($total * Config::$cancelCharge[$days] * $cent) / $cent;
                    //$cancelCharge = $charge->currency == 'yen' ? floor($total * Config::$cancelCharge[$days]) : floor($total * Config::$cancelCharge[$days] * 100) / 100;
                } // 固定金額(表引き)
                elseif ($settingCancel->charge == 2) {
                    $cancelCharge = Config::$cancelCharge[$days];
                }
            }

            // キャンセル料金の消費税計算
            if (0 < $settingCancel->tax && 0 < $cancelCharge) {
                $tax = $this->computeTax($cancelCharge);
            }
        }

        // 総合計
        $cancelTotal = $cancelCharge + $tax;

        $cancel = array(
            'total' => $total,
            'cancelCharge' => $cancelCharge,
            'taxRate' => $settingPayment->tax,
            'tax' => $tax,
            'consumption' => $settingPayment->consumption,
            'cancelTotal' => $cancelTotal,
            'currency' => $settingPayment->currency,
            'cancelTime' => $cancelTime,
        );

        $this->setPayment('cancel', $cancel);

        return $cancel;
    }

    /**
     * 決済受付可能か確認する(受付締切までの日数を戻す)
     */
    public function paymentAvailable()
    {
        $settingPayment = Config::getSetting('payment');

        // 決済機能を利用しない
        if ($settingPayment->clearance <= 0) {
            return false;
        }

        // 貸渡日
        $rentTime = $this->rent_start - $this->rent_start % 86400;

        // 決済受付開始日
        if ($settingPayment->assigned == 0) {
            $assigned = strtotime(substr($this->created, 0, 10));
        } else {
            $assigned = $rentTime - $settingPayment->assigned * 86400;
        }

        // 決済受付終了
        $termEnd = $assigned + $settingPayment->term * 86400;

        // 決済受付期限
        $limit = $rentTime - $settingPayment->limit * 86400;

        // 受付期限日
        $limitTime = $limit < $termEnd ? $limit : $termEnd;

        // 現在
        $todayTime = current_time('timestamp');
        $todayTime -= $todayTime % 86400;

        if ($todayTime < $assigned || $limitTime < $todayTime) {
            return false;
        }

        return array(
            'assigned' => $assigned,
            'limit' => $limitTime,
            'remain' => ceil(($limitTime - $assigned) / 86400),
        );
    }

}