<?php
/**
 * MTS Car Booking レンタカーボックス表示処理モジュール
 *
 * @Filename    RentalBox.php
 * @Author      S.Hayashi
 * @Code        2018-11-30 Ver.1.0.0
 */

namespace MTSCarBookingTrial;

use MTSCarBookingTrial\models\Vehicle;
use MTSCarBookingTrial\views\ParamAdmin;
use MTSCarBookingTrial\lang\Translation;

class RentalBox
{
    // ショートコードパラメータ初期値
    private $init = array(
        'vid' => array(),
        'gid' => array(),
        'pid' => 0,
        'tno' => 0,
        'name' => 0,                // 車両名h3,caption表示
        'link' => 1,                // 予約ページリンク
        'img' => 0,                 // 車両画像表示(-1:非表示)
        'detail' => array(),
        'charge' => array(),
        'lang' => Config::LANG,
    );

    // ショートコードのパラメータ
    private $data = array();

    // 予約ページリンク
    private $bookingUrl = '';

    // カースペック項目
    private $details = array();

    // 車両データ
    private $vehicles = array();

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->data = $this->init;

        // 予約カレンダー表示リンクURL
        $this->bookingUrl = get_permalink(get_page_by_path(CarBooking::BOOKING_PAGE));

        // カースペック項目一覧
        $this->init['detail'] = array_keys(ParamAdmin::$vehicleDetail);

    }

    /**
     * レンタカーボックスを出力する
     */
    public function rendering($atts)
    {
        $view = '';

        // 表示対象車両データを取得する
        $this->_initParam($atts);

        // 車両データを表示する
        foreach ($this->vid as $vehicleId) {
            $view .= $this->_showVehicle($vehicleId);
        }

        return $view;
    }

    // ショートコードパラメータから表示車両データを準備する
    private function _initParam($atts)
    {
        foreach ($this->init as $key => $val) {
            $this->{$key} = $val;
            if (array_key_exists($key, $atts)) {
                switch ($key) {
                    case 'vid':
                        $this->{$key} = explode(',', $atts[$key]);
                        break;
                    case 'gid':
                        foreach (explode(',', $atts[$key]) as $termId) {
                            $this->data['vid'] += array_keys(Vehicle::listVehicleByType($termId));
                        }
                        break;
                    case 'pid':
                    case 'tno':
                    case 'name':
                    case 'link':
                        $this->{$key} = intval($atts[$key]);
                        break;
                    case 'detail':
                        $this->detail = explode(',', $atts[$key]);
                        break;
                    case 'lang':
                        $this->{$key} = $atts[$key];
                        break;
                }
            }
        }
    }

    // 車両データを表示する
    private function _showVehicle($vehicleId)
    {
        if (isset($this->vehicles[$vehicleId])) {
            $oVehicle = $this->vehicles[$vehicleId];
        }

        else {
            $oVehicle = Vehicle::readVehicle($vehicleId);
            if (0 < $oVehicle->postId) {
                $this->vehicles[$vehicleId] = $oVehicle;
            } else {
                return 'NOT_FOUND_VEHICLE';
            }
        }

        $vars = $this->mkVar($oVehicle);

        $template = $this->_template($this->tno, $oVehicle);

        // 表示HTMLを戻す
        return str_replace(array_keys($vars), array_values($vars), $template);
    }

    // 埋め込み変数の値を生成する
    protected function mkVar(Vehicle $oVehicle)
    {
        // 車両データ情報
        $vars = array(
            '%VEHICLE_ID%' => $oVehicle->postId,
            '%VEHICLE_NAME%' => $oVehicle->name->{$this->lang},
            '%BOOKING_CAR%' => add_query_arg(array('vid' => $oVehicle->postId), $this->bookingUrl),
            '%FUEL%' => Translation::$shortCode[strtoupper($oVehicle->charge->fuel)][$this->lang],
            '%FUEL_KM%' => $oVehicle->charge->fuel_km,
            '%IMAGE_0%' => $oVehicle->image[0],
            '%IMAGE_1%' => $oVehicle->image[1],
        );

        // 車両詳細情報
        $detailVar = array();
        foreach ($oVehicle->detail as $idx => $val) {
            $detailVar[sprintf('%%%s%%', strtoupper($idx))] = $val;
        }

        // 車両料金情報
        $chargeVar = array();
        if (0 < $oVehicle->charge->charge_id) {
            $oCharge = $oVehicle->oCharge;
            $chargeVar = array(
                '%CHARGE_CURRENCY%' => Translation::$fCharge[strtoupper($oCharge->charge->currency)][$this->lang],
                '%CHARGE_PRICE%' => number_format($oCharge->charge->regular['fee']),
                '%CHARGE_LIMITED%' => number_format($oCharge->charge->limited['fee']),
            );

            // テーブル通常料金
            foreach ($oCharge->regular as $idx => $row) {
                $chargeVar["%REG_BYH_{$idx}%"] = intval($row['time'] / 3600);
                $chargeVar["%REG_BYD_{$idx}%"] = intval($row['time'] / 86400);
                $chargeVar["%REG_FEE_{$idx}%"] = number_format($row['price']);
            }

            // テーブル特別料金
            foreach ($oCharge->limited as $idx => $row) {
                $chargeVar["%LIM_BYH_{$idx}%"] = intval($row['time'] / 3600);
                $chargeVar["%LIM_BYD_{$idx}%"] = intval($row['time'] / 86400);
                $chargeVar["%LIM_FEE_{$idx}%"] = number_format($row['price']);
            }
        }

        // ユーザー定義変数を付加する
        $userVar = apply_filters('mtsrcb_rental_box_var', array(), $oVehicle);

        return $vars + $chargeVar + $detailVar + $userVar;
    }

    // 表示HTMLテンプレートソースを戻す
    private function _template($no, $oVehicle)
    {
        // 投稿のテンプレートがあればそれを戻す
        if (0 < $this->pid) {
            return get_post_field('post_content', $this->pid, 'raw');
        }

        // レンタカースペック表示
        if ($no == 0) :
            $rowspan = count($this->data['detail']) + 1;
            ob_start();
            $row = 0;
?>
        <div class="rental-car-spec">
            <?php echo $this->_disp_name() ?>
            <table class="car-spec">
                <?php if (0 <= $this->img) {
                    echo sprintf('<tr class="row-img"><td rowspan="%d" class="vehicle-image"><img src="%%IMAGE_%d%%"></td></tr>',
                    $rowspan, $this->img);
                } ?>
                <?php foreach ($this->detail as $key) : ?><tr class="row-spec <?php echo sprintf('row-%s', $row % 2 ? 'odd' : 'even') ?>">
                    <th><?php echo $this->lang === Config::LANG ? ParamAdmin::$vehicleDetail[$key] : ucfirst($key) ?></th>
                    <td><?php echo sprintf('%%%s%%', strtoupper($key)) ?></td>
                </tr><?php endforeach; ?>
            </table>
            <?php echo $this->_disp_link() ?>
        </div>

<?php
        // レンタカー料金表示
        elseif ($no == 1) :
            if (!$oVehicle->oCharge) {
                return 'UNSET_CHARGE_TABLE';
            }

            $oCharge = $oVehicle->oCharge;
            $feerows = count($oCharge->regular);
            $currency = strtoupper($oCharge->charge->currency);
            ob_start();
?>
        <div class="rental-car-charge">
            <?php echo $this->_disp_name() ?>
            <table class="car-charge">
                <?php if (0 <= $this->img) {
                    echo sprintf('<tr class="row-img"><td colspan="2" class="vehicle-image"><img src="%%IMAGE_%d%%"></td></tr>',
                        $this->img);
                } ?>
                <?php for ($i = 0; $i < $feerows; $i++) :
                    $time = intval($oCharge->regular[$i]['time'] / 86400);
                    if ($time <= 1) {
                        $time = intval($oCharge->regular[$i]['time'] / 3600);
                        $unit = $this->lang === 'ja' ? '時間' : 'hour';
                    } else {
                        $unit = $this->lang === 'ja' ? '日' : 'day';
                    }
                    if ($this->lang !== 'ja' && 1 < $time) {
                        $unit .= 's';
                    }

                    $oddEven = 0 < $i % 2 ? 'odd' : 'even';

                    echo sprintf('<tr class="row-%s"><th><span class="rental-time">%s<span class="time-unit">%s</span></th>',
                        $oddEven, $time, $unit);
                    echo sprintf('<td><span class="time-fee">%s</span><span class="charge-currency">%s</span></td></tr>',
                        number_format($oCharge->regular[$i]['price']), Translation::$fCharge[$currency][$this->lang]);
                    echo "\n";
                endfor; ?>

                <tr>
                    <td class="car-detail" colspan="2">
                        <?php foreach ($this->detail as $key) {
                            if ($oVehicle->detail->{$key}) {
                                echo sprintf('<span class="detail-item %s">%s</span>', $key, $oVehicle->detail->{$key});
                            }
                        } ?></td>
                </tr>
            </table>
            <?php echo $this->_disp_link() ?>
        </div>

<?php
        endif;

        return ob_get_clean();
    }

    // 車両名表示
    private function _disp_name()
    {
        if (0 < $this->name) {
            return '<h3 class="car-name">%VEHICLE_NAME%</h3>';
        }

        return '';
     }

    // 予約ページリンク表示
    private function _disp_link()
    {
        if ($this->link <= 0) {
            return '';
        }

        return sprintf('<div class="blooking-link"><a href="%s">%s</a></div>',
            '%BOOKING_CAR%', Translation::$fButton['BOOKING'][$this->lang]);
    }




    /**
     * プロパティから読み出す
     */
    public function __get($key)
    {
        if (array_key_exists($key, $this->data)) {
            if (is_array($this->data[$key])) {
                return (object) $this->data[$key];
            }
            return $this->data[$key];
        }

        if (isset($this->$key)) {
            return $this->$key;
        }

        $trace = debug_backtrace();
        trigger_error(sprintf(
            "Undefined property: '%s&' in %s on line %d, E_USER_NOTICE",
            $key, $trace[0]['file'], $trace[0]['line']
        ));

        return null;
    }

    /**
     * プロパティをセットする
     */
    public function __set($key, $value)
    {
        if (array_key_exists($key, $this->data)) {
            //if (is_array($value)) {
            //    foreach ($value as $item => $val) {
            //        $this->data[$key][$item] = $val;
            //    }
            //} else {
                $this->data[$key] = $value;
            //}
        } else {
            $this->$key = $value;
        }

        return $value;
    }

}
