<?php
/**
 * MTS Car Booking フロントエンド予約カレンダー表示処理
 *
 * @Filename    BookingCalendar.php
 * @Author      S.Hayashi
 * @Code        2019-01-11 Ver.1.0.0
 */
namespace MTSCarBookingTrial;

use MTSCarBookingTrial\models\Calendar;
use MTSCarBookingTrial\models\ShopCalendar;
use MTSCarBookingTrial\models\Schedule;
use MTSCarBookingTrial\views\BookingCalendarView;

class BookingCalendar
{
    // 表示使用言語
    public $lang = Config::LANG;

    // メッセージ
    public $msgCode = '';
    public $msgSub = '';
    public $errflg = false;

    // カレンダー表示データ定義
    private static $daily = array(
        'status' => 'closed',
        'classes' => array(),
        'memo' => '',
    );

    // 各種設定
    private $settingReserve = null;

    // カレンダーデータ
    private $oCalendar = null;

    // 予約受付期間
    private $acceptClose = 0;               // この日まで(直近)
    private $acceptOpen = 0;                // この日から(将来)

    // 予約受付月
    private $linkLast = 0;                  // この月から(直近)
    private $linkFuture = 0;                // この月まで(将来)

    private $shopCalendar = array();        // 指定月の営業カレンダー
    private $vehicleSchedule = array();     // 指定月指定車両の予約スケジュール
    private $rentalCalendar = array();      // １カ月カレンダー表示データ

    /**
     * Constructor
     */
    public function __construct(Calendar $oCalendar)
    {
        $this->oCalendar = new Calendar;

        // 各種設定の予約条件を取得する
        $this->settingReserve = Config::getSetting('reserve');

        // 前月リンク有効期限
        $this->oCalendar->linkLast = $oCalendar->linkLast;

        // 翌月リンク有効期限
        $this->oCalendar->linkFuture = $oCalendar->linkFuture;

        // 直近の予約受付有効日
        $this->acceptClose = $oCalendar->todayTime + $this->settingReserve->close * 86400;

        // 将来の予約受付有効日
        $this->acceptOpen =  $oCalendar->todayTime + $this->settingReserve->open * 86400;
    }

    /**
     * 予約カレンダーのビューを戻す
     */
    public function calendar($vehicleId, $monthTime, $lang)
    {
        $this->lang = $lang;

        // 指定月が表示期間内かチェックする
        if ($monthTime < $this->oCalendar->linkLast) {
            $monthTime = $this->oCalendar->linkLast;
        } elseif ($this->oCalendar->linkFuture < $monthTime) {
            $monthTime = $this->oCalendar->linkFuture;
        }

        $calendar = '';

        // 表示月数のカレンダー出力を用意する
        for ($i = 0; $i < $this->settingReserve->months && $monthTime <= $this->oCalendar->linkFuture; $i++) {
            // 表示月のカレンダーオブジェクト設定
            $this->oCalendar->setCalendar($monthTime);

            // 営業カレンダーを取得する
            if ($this->_getShopCalendar()) {
                // 指定車両の指定月の予約スケジュールを取得する
                $this->_getVehicleSchedule($vehicleId);
                // 予約カレンダー表示データを準備する
                $this->_setupCalendarData();

                // カレンダー表示ビューモジュールを用意する
                $oCalendarView = new BookingCalendarView($this->oCalendar, $this->rentalCalendar, $lang);

                $calendar .= $oCalendarView->bookingCalendar($this->settingReserve->months);

                $monthTime = $this->oCalendar->nextTime;
            }
        }

        return $calendar;
    }

    // 指定月の営業カレンダーを取得する
    private function _getShopCalendar()
    {
        $oShopCalendar = new ShopCalendar;

        if (!$oShopCalendar->readShopCalendar($this->oCalendar->monthTime)) {
            return $this->_setError('SCHEDULE_NOT_DETERMINED');
        }

        $shopMonth =  $oShopCalendar->getShopMonth($this->oCalendar->monthTime);

        $shopCalendar = array();

        // 営業カレンダー１カ月に展開する
        $datetime = $this->oCalendar->monthTime;
        foreach ($shopMonth as $shopData) {
            $shopCalendar[$datetime] = $shopData;
            $datetime += 86400;
        }

        $this->shopCalendar = $shopCalendar;

        return true;
    }

    // 指定車両の指定月の予約データを取得する
    private function _getVehicleSchedule($vehicleId)
    {
        // 車両スケジュールデータを取得する
        $schedules = Schedule::findRentalVehicle($this->oCalendar->monthTime, $this->oCalendar->nextTime);

        // 車両１カ月スケジュールをクリアする
        for ($datetime = $this->oCalendar->monthTime; $datetime < $this->oCalendar->nextTime; $datetime += 86400) {
            $this->vehicleSchedule[$datetime] = false;
        }

        if (!isset($schedules[$vehicleId])) {
            return;
        }

        // レンタルスケジュールデータを展開する
        foreach ($schedules[$vehicleId] as $schedule) {
            // 貸渡日0時0分
            $rentStart = $schedule['rent_start'] - $schedule['rent_start'] % 86400;
            // 返却日翌日0時0分
            $rentEnd = $schedule['rent_end'] + (86400 - $schedule['rent_end'] % 86400);
            // １影月スケジュールに貸し出しをセットする
            for ($renttime = $rentStart; $renttime < $rentEnd; $renttime += 86400) {
                if ($this->oCalendar->monthTime <= $renttime && $renttime < $this->oCalendar->nextTime) {
                    $this->vehicleSchedule[$renttime] = true;
                }
            }
        }
    }

    // 予約カレンダー表示データを準備する
    private function _setupCalendarData()
    {
        // 季節営業期間
        $on_time = strtotime(sprintf('%s-%s', $this->oCalendar->year, $this->settingReserve->on_season));
        $off_time = strtotime(sprintf('%s-%s', $this->oCalendar->year, $this->settingReserve->off_season));

        // １カ月のカレンダー表示データを用意する
        for ($datetime = $this->oCalendar->monthTime; $datetime < $this->oCalendar->nextTime; $datetime += 86400) {
            // 車両の予約状況をセットする
            $status = $this->vehicleSchedule[$datetime] ? 'booked' : 'available';

            // 営業カレンダー休業日をセットする
            $shopDay = $this->shopCalendar[$datetime];
            if (!$shopDay['open']) {
                $status = 'closed';
            }

            // 季節営業の状態をセットする
            if ($on_time < $off_time) {
                // 夏営業チェック
                if ($datetime < $on_time || $off_time < $datetime) {
                    $status = 'closed';
                }
            } elseif ($on_time > $off_time) {
                // 冬営業チェック
                if ($on_time < $datetime && $datetime < $off_time) {
                    $status = 'closed';
                }
            }

            // 予約受付期間外なら休業日をセットする
            if ($datetime < $this->acceptClose || $this->acceptOpen < $datetime) {
                $status = 'closed';
            }

            // 当該日の表示データ
            $rental = self::$daily;
            $rental['status'] = $status;
            $rental['classes'] = $this->_getClasses($datetime, $shopDay, $status);
            $rental['memo'] = isset($shopDay['memo'][$this->lang]) ? $shopDay['memo'][$this->lang] : '';

            $this->rentalCalendar[$datetime] = $rental;
        }
    }

    // カレンダー内指定クラスデータ
    private function _getClasses($datetime, $shopDay, $status)
    {
        $classes = array();

        $classes[] = sprintf('day-%d', date('j', $datetime));

        $classes[] = strtolower(date('D', $datetime));

        $classes[] = $status === 'close' ? 'shop-close' : $status;

        // 営業カレンダークラス設定
        if ($shopDay['class'] != '') {
            $classes = array_merge($classes, explode(' ', $shopDay['class']));
        }

        // 本日であればセットする
        if ($datetime == $this->oCalendar->todayTime) {
            $classes[] = 'today';
        }

        return $classes;
    }

    // エラーセット
    private function _setError($code, $sub='')
    {
        $this->msgCode = $code;
        $this->msgSub = $sub;
        $this->errflg = true;
        return false;
    }

}
