<?php
/**
 * MTS Car Booking 予約者入力・編集処理ビュー
 *
 * @Filename    CustomerView.php
 * @Author      S.Hayashi
 * @Code        2018-08-30 Ver.1.0.0
 */
namespace MTSCarBookingTrial\views;

use MTSCarBookingTrial\Config;
use MTSCarBookingTrial\lang\Translation;
use MTSCarBookingTrial\models\Customer;

class CustomerView
{
    const PAGE_NAME = 'mts-custom';
    const LOADER_FILE = 'img/ajax-loader-mini.gif';

    public static $zipJS = array(
        0 => array(),
        1 => array(
            'mts_assistance' => 'js/mts-assistance.js',
        ),
        2 => array(
            'post_codejp' => 'https://postcode-jp.com/js/postcodejp.js',
            'mts_postcodejp' => 'js/mts-postcodejp.js',
        ),
    );


    private $iconUrl = '';

    private $setting = array();
    private $oCustomer = null;

    private $adminFlg = false;
    private $lang = Config::LANG;

    private $translation = array();
    private $oHelper;

    public $error = array();

    /**
     * CustomerView constructor
     */
    public function __construct()
    {
        $this->translation = Translation::$customer;
        $this->oHelper = FormHelper::getInstance();

        // AJAXローディングアイコン画像のURL
        $this->iconUrl = plugins_url(self::LOADER_FILE, dirname(__FILE__));
    }

    /**
     * 郵便番号検索入力準備
     */
    public static function enqueuePostcode($zipSearch)
    {
        if (empty(self::$zipJS[$zipSearch])) {
            return;
        }

        $dir = dirname(__FILE__);

        foreach (self::$zipJS[$zipSearch] as $key => $url) {
            if (!preg_match('/^http.*$/', $url)) {
                $url = plugins_url($url, $dir);
            }

            wp_enqueue_script($key, $url, array('jquery'));
        }
    }

    /**
     * 予約者情報入力フロントエンド
     */
    public function formView(Customer $oCustomer, $setting, $tableId='mts')
    {
        $this->lang = $oCustomer->lang;
        $this->setting = $setting;
        $this->oCustomer = $oCustomer;

        $this->_entryView($tableId);
    }

    /**
     * 予約者情報入力管理画面
     */
    public function adminView(Customer $oCustomer, $setting, $tableId='admin')
    {
        $this->adminFlg = true;
        $this->setting = $setting;
        $this->oCustomer = $oCustomer;

        $this->_entryView($tableId);
    }

    private function _entryView($tableId)
    {
        $oCustomer = $this->oCustomer;
        $this->error = $oCustomer->error;

?>
        <table id="<?php echo $tableId ?>-customer-entry" class="mts-form-table">
        <?php foreach ($this->setting as $item => $usage) {
            if ($usage == 0) {
                continue;
            }
            switch ($item) {
                case 'seimei':
                    $this->_nameView($item);
                    break;
                case 'furigana':
                    $this->_kanaView($item);
                    break;
                case 'company':
                case 'email':
                case 'email2':
                    $this->_textView($item, 'fat');
                    break;
                case 'jusho':
                    $this->_addressView($item);
                    break;
                case 'country':
                case 'tel':
                case 'mobile':
                    $this->_textView($item, 'middle-small');
                    break;

                case 'postcode':
                case 'gender':
                case 'birthday':
                    call_user_func(array($this, "_{$item}View"), $item);
                    break;

                case 'key':
                case 'api_key':
                    break;

                default:
                    if (method_exists($this, "_{$item}View")) {
                        $val = $oCustomer->$item;
                        $error = isset($oCustomer->error[$item]) ? $oCustomer->error[$item] : '';
                        apply_filters('mts_customer_item', '', array($oCustomer->items->$item, $val, $error));
                    }
                    break;
            }
        } ?>
        </table>

<?php
        //</div>
    }

    // テキスト入力項目の出力
    private function _textView($item, $size)
    {
?>
        <tr>
            <th><?php
                echo sprintf('<label for="customer-%s">%s%s</label>',
                    $item, $this->translation[$item][$this->lang], $this->_strRequired($item))
            ?></th>
            <td><?php
                echo sprintf('<input id="customer-%s" type="text" class="mts-%s" name="customer[%s]" value="%s">',
                    $item, $size, $item, esc_html($this->oCustomer->$item));
                $this->_outItemError($item);
                $this->_outDescription($item);
            ?></td>
        </tr>

<?php
    }

    // 名前入力項目の出力
    private function _nameView($item)
    {
        $line = Translation::$nameLine[$this->lang];

?>
        <tr>
            <th><?php
                echo sprintf('<label for="customer-%s">', $line[0]);
                echo $this->translation[$item][$this->lang];
                echo $this->_strRequired($item) ?></label></th>
            <td><?php foreach($line as $column) {
                echo '<div class="customer-name-box">';
                echo sprintf('<label for="customer-%s" class="customer-name-label">%s</label>', $column, $this->translation[$column][$this->lang]);
                echo sprintf('<input id="customer-%s" class="customer-name" type="text" name="customer[%s]" value="%s">',
                    $column, $column, esc_html($this->oCustomer->{$column}));
                echo "</div>\n";
            }
            $this->_outItemError($item);
            $this->_outDescription($item);
            ?></td>
        </tr>

<?php
    }

    // フリガナ入力項目の出力
    private function _kanaView($item)
    {
        $line = Translation::$nameLine[$this->lang];

?>
        <tr>
            <th><?php
                echo sprintf('<label for="customer-%s-kana">', $line[0]);
                echo $this->translation[$item][$this->lang];
                echo $this->_strRequired($item) ?></label></th>
            <td><?php foreach ($line as $column) {
                $valname = $column . '_kana';
                echo '<div class="customer-name-box">';
                echo sprintf('<label for="customer-%s-kana" class="customer-name-label">%s</label>', $column, $this->translation[$valname][$this->lang]);
                echo sprintf('<input id="customer-%s-kana" class="customer-name" type="text" name="customer[%s_kana]" value="%s">',
                    $column, $column, esc_html($this->oCustomer->{$valname}));
                echo "</div>\n";
            }
            $this->_outItemError($item);
            $this->_outDescription($item);
            ?></td>
        </tr>

<?php
    }

    // 郵便番号項目の出力
    private function _postcodeView($item)
    {
?>
        <tr>
            <th>
                <label for="customer-postcode"><?php echo $this->translation[$item][$this->lang] ?></label>
            </th>
            <td><?php
                    echo $this->oHelper->textBox('customer-postcode', 'mts-char8', 'customer[postcode]', $this->oCustomer->postcode);
                    if ($this->lang === 'ja' && $this->setting['zip_search'] && $this->setting['api_key'] != '') {
                        echo sprintf('<button id="mts-postcode-button" type="button" class="button" onclick="mts_assistance.findByPostcode(\'%s\', \'customer-postcode\', \'customer-pref\', \'customer-city\', \'customer-town\')">検索</button>',
                            $this->setting['api_key']);
                        echo sprintf('<img id="customer-postcode-loading" src="%s" style="display:none" alt="Loading...">', $this->iconUrl);
                    }
                    $this->_outItemError($item);
                    $this->_outDescription($item); ?>
            </td>
        </tr>

<?php
    }

    // 住所入力項目の出力
    private function _addressView($item)
    {
        $line = Translation::$addressLine[$this->lang];
?>
        <tr>
            <th><label for="customer-pref"><?php
                echo $this->translation[$item][$this->lang];
                    $this->_strRequired($item) ?></label></th>
            <td>
                <dl><?php foreach ($line as $column) {
                    echo sprintf('<dt class="customer-header"><label for="customer-%s">%s</label></dt>',
                        $column, $this->translation[$column][$this->lang]);
                    echo "<dd>";
                    if ($column == 'pref' && $this->lang == 'ja') {
                        echo $this->oHelper->selectPref('customer-pref', 'mts-select', 'customer[pref]', esc_html($this->oCustomer->pref));
                    } else {
                        echo $this->oHelper->textBox("customer-{$column}", 'mts-fat', "customer[{$column}]", esc_html($this->oCustomer->{$column}));
                    }
                    $this->_outDescription($column);
                    echo "</dd>\n";
                }

                echo "</dl>\n";
                $this->_outItemError($item);
                $this->_outDescription($item);
                ?>

            </td>
        </tr>

<?php
    }

    // 性別入力項目の出力
    private function _genderView($item)
    {
        $line = Translation::$genderLine[$this->lang];

?>
        <tr>
            <th><?php
                echo $this->translation[$item][$this->lang];
                $this->_strRequired($item) ?></th>
            <td><?php foreach ($line as $column) {
                echo sprintf('<input id="customer-%s" class="gender-radio" type="radio" name="customer[gender]" value="%s"%s>',
                    $column, $column, $this->oHelper->setChecked($this->oCustomer->gender, $column));
                echo sprintf('<label for="customer-%s" class="gender-label">%s</label>', $column, $this->translation[$column][$this->lang]);
                }
                $this->_outItemError($item);
                $this->_outDescription($item); ?></td>
        </tr>
<?php
    }

    // 生年月日入力項目の出力
    private function _birthdayView($item)
    {
        if ($this->lang != 'ja') {
            $line = Translation::$dateLine[$this->lang];

            $aDate = explode('-', $this->oCustomer->birthday == '' ? '0-0-0' : $this->oCustomer->birthday);
            $year = $aDate[0];
            $month = $aDate[1];
            $day = $aDate[2];

            $startYear = intval(date_i18n('Y'));
            $endYear = $startYear - 100;
        }

?>
        <tr>
            <th><?php
                echo $this->translation[$item][$this->lang];
                $this->_strRequired($item); ?></th>
            <td><?php if ($this->lang == 'ja') {
                echo $this->oHelper->selectBirthday('customer-birthday', 'mts-birthday', 'customer[birthday]', $this->oCustomer->birthday);
            } else {
                foreach ($line as $column) {
                    switch ($column) {
                        case 'year':
                            echo $this->oHelper->selectYear('customer-birthday-year', 'mts-birthday', 'customer[birthday][year]', $year, $startYear, $endYear, '');
                            break;
                        case 'month':
                        case 'day':
                            echo $this->oHelper->selectMonth("customer-birthday-{$column}", 'mts-birthday', "customer[birthday][{$column}]", $$column, '<span class="date-slash">/</span>');
                            break;
                    }
                }
            }
            $this->_outItemError($item);
            $this->_outDescription($item); ?></td>
        </tr>
<?php
    }

    // 必須入力表示
    private function _strRequired($item)
    {
        $required = '';

        if ($this->setting[$item] == 2) {
            $required = Translation::$others['required'][$this->lang];
        }

        return $required;
    }

    // エラー項目表示
    private function _outItemError($item)
    {
        if (array_key_exists($item, $this->error)) {
            echo sprintf('<div class="mts-error">%s</div>', Translation::$cError[$this->error[$item]][$this->lang]);
        }
    }

    // 項目説明表示
    private function _outDescription($item)
    {
        if (!empty(Translation::$cDescription[$item][$this->lang])) {
            echo sprintf('<div class="description">%s</div>', Translation::$cDescription[$item][$this->lang]);
        }
    }

    /**
     * 予約者情報確認フロントエンド
     */
    public function confirmationView($oCustomer, $setting)
    {
        $this->lang = $oCustomer->lang;
        //$this->setting = $setting;
        //$this->oCustomer = $oCustomer;

?>
        <table id="mts-customer-confirm" class="mts-form-table">
            <?php foreach ($setting as $item => $usage) {
                if ($usage == 0) {
                    continue;
                }
                switch ($item) {
                    case 'seimei':
                        $this->_itemName($item, $oCustomer->sei, $oCustomer->mei);
                        break;
                    case 'furigana':
                        $this->_itemName($item, $oCustomer->sei_kana, $oCustomer->mei_kana);
                        break;
                    case 'company':
                    case 'postcode':
                    case 'country':
                    case 'email':
                    case 'tel':
                    case 'mobile':
                    case 'gender':
                    case 'birthday':
                    $this->_itemRow($item, $oCustomer->{$item});
                        break;
                    case 'jusho':
                        $this->_itemAddress($item, $oCustomer->pref, $oCustomer->city, $oCustomer->town, $oCustomer->building);
                        break;
                    case 'email2':
                    case 'key':
                    case 'google_key':
                        break;
                    default:
                        if (method_exists($this, "{$item}View")) {
                            call_user_func(array($this, "{$item}View"), $oCustomer->$item);
                        }
                        break;
                }
            } ?>
        </table>

<?php
    }

    // 名前、フリガナ
    private function _itemName($item, $sei, $mei)
    {
        $line = Translation::$nameLine[$this->lang];
?>
        <tr>
            <th class="customer-view-title"><?php echo Translation::$customer[$item][$this->lang] ?></th>
            <td><?php foreach ($line as $column) {
                if ($$column != '') {
                    echo sprintf('<span class="customer-name-item">%s</span>', esc_html($$column));
                }
        } ?></td>
        </tr>
<?php
    }

    // 住所
    private function _itemAddress($item, $pref, $city, $town, $building)
    {
        $line = Translation::$addressLine[$this->lang];
?>
        <tr>
            <th class="customer-view-title"><?php echo Translation::$customer[$item][$this->lang] ?></th>
            <td><?php foreach ($line as $column) {
                if ($$column != '') {
                    echo sprintf('<span class="customer-address-item">%s</span>', esc_html($$column));
                }
            } ?></td>
        </tr>

<?php
    }

    // その他項目
    private function _itemRow($item, $val)
    {
?>
        <tr>
            <th class="customer-view-title"><?php echo Translation::$customer[$item][$this->lang] ?></th>
            <td><?php
                switch ($item) {
                    case 'gender':
                        if (in_array($val, array('male', 'female'))) {
                            echo Translation::$customer[$val][$this->lang];
                        }
                        break;
                    case 'birthday':
                        $this->_outBirthday($val);
                        break;
                    default:
                        echo esc_html($val);
                }
            ?></td>
        </tr>
<?php
    }

    // 誕生日表示
    private function _outBirthday($ymdStr)
    {
        if (empty($ymdStr) || $ymdStr == '0000-00-00') {
            return;
        }

        $line = Translation::$dateLine[$this->lang];

        $ymd = explode('-', $ymdStr);

        foreach ($line as $column) {
            if ($column == 'year') {
                $birth = $ymd[0];
                $suffix = $this->lang == 'ja' ? '年' : '';
            } elseif ($column == 'month') {
                if ($this->lang) {
                    $birth = sprintf('%d', $ymd[1]);
                    $suffix = '月';
                } else {
                    $birth = Translation::$monthF[$ymd[1] - 1];
                    $suffix = '';
                }
            } elseif ($column == 'day') {
                $birth = sprintf('%d', $ymd[2]);
                $suffix = $this->lang == 'ja' ? '日' : '';
            }
            echo sprintf('<span class="customer-birthday-item">%d%s</span>', $birth, $suffix);
        }
    }

}
