<?php
/**
 * MTS Car Booking カレンダー表示ビュー
 *
 * @Filename    CalendarView.php
 * @Author      S.Hayashi
 * @Code        2018-07-31 Ver.1.0.0
  */
namespace MTSCarBookingTrial\views;

use MTSCarBookingTrial\models\Calendar;

class CalendarView
{
    static public $weekClass = array('sun', 'mon', 'tue', 'wed', 'thu', 'fri', 'sat');

    private $data = array(
        'anchor' => '',
        'tableId' => '',
        'tableClass' => '',
        'captionUse' => true,       // キャプションを表示
        'captionSide' => 'top',     // キャプション表示位置
        'captionTitle' => true,     // キャプションにカレンダー月を表示
        'captionLink' => false,     // キャプションに月切替リンクを表示
        'thead' => true,            // テーブル上段に曜日タイトルを表示
        'tfoot' => false,           // テーブル下段に曜日タイトルを表示
        'linkLast' => 0,            // 前月リンク有効限time
        'linkFuture' => 0,          // 来月リンク有効限time
        'prev' => '＜',
        'prev_prev' => '＜＜',
        'next' => '＞',
        'next_next' => '＞＞',

        'week' => array('日', '月', '火', '水', '木', '金', '土'),
        'month' => 'n月',
        'lang' => 'ja',
        'linkUrl' => '',
    );

    // TDセル個別表示の外部コール関数
    public $callTdClass = null;         // tdタグにclassを追加
    public $callDayNumber = null;       // 日付行に追加
    public $callDayContent = null;      // カレンダー日付内コンテンツ

    // TDセル全体表示の外部コール関数
    public $callTdCell = null;

    // TDセル埋め草の外部コール関数
    public $callTdNoDay = null;

    private $oCalendar;

    /**
     * Constructor
     */
    public function __construct()
    {
        if ($this->linkUrl == '') {
            $this->linkUrl = $_SERVER['REQUEST_URI'];
        }
    }

    /**
     * １カ月カレンダーの表示
     */
    public function monthCalendar(Calendar $oCalendar)
    {
        $this->oCalendar = $oCalendar;

        // カレンダー先頭のUnix Time
        $startTime = $oCalendar->monthTime - (date('w', $oCalendar->monthTime) - $oCalendar->startOfWeek + 7) % 7 * 86400;

        ob_start();

        echo '<table' . $this->_tableProperty() . ">\n";

        if ($this->captionUse) {
            $this->outCaption($oCalendar);
        }

        if ($this->thead) {
            $this->_outHeader('thead');
        }

        if ($this->tfoot) {
            $this->_outHeader('tfoot');
        }

        // カレンダー先頭の時間をセットして１カ月表示
        for ($bx = 0, $datetime = $startTime; $bx < 42; $bx++, $datetime += 86400) {

            if ($bx % 7 == 0) {
                if (0 < $bx) {
                    echo "</tr>\n";
                }

                // 月末日を越えた場合は表示を終了する
                if ($oCalendar->nextTime <= $datetime) {
                    break;
                }

                echo sprintf('<tr class="week-row-%d">', intval($bx / 7) + 1);
            }

            // 曜日No
            $weekNo = ($oCalendar->startOfWeek + $bx) % 7;

            // カレンダー日付処理
            if ($oCalendar->monthTime <= $datetime && $datetime < $oCalendar->nextTime) {
                $this->_outDate($datetime);
            }

            // カレンダー日付外処理
            else {
                $this->_outNoDate($weekNo);
            }
        }

        echo "</table>\n";

        return ob_get_clean();
    }

    // テーブルセグメント属性
    private function _tableProperty()
    {
        $attr = '';

        if ($this->tableId != '') {
            $attr .= sprintf(' id="%s"', $this->tableId);
        }

        if ($this->tableClass != '') {
            $attr .= sprintf(' class="%s"', $this->tableClass);
        }

        return $attr;
    }

    // カレンダーの曜日行表示
    private function _outHeader($pos)
    {
        echo sprintf('<%s><tr class="week-row-%s">', $pos, $pos);

        for ($i = 0, $weekNo = $this->oCalendar->startOfWeek; $i < 7; $i++, $weekNo = ++$weekNo % 7) {
            echo sprintf('<th class="%s">%s</th>', self::$weekClass[$weekNo], $this->week[$weekNo]);
        }

        echo sprintf('</tr></%s>', $pos) . "\n";
    }

    // 日付表示
    private function _date($format, $time)
    {
        if ($this->lang === 'ja') {
            return date_i18n($format, $time);
        }

        return date($format, $time);
    }

    /**
     * キャプション表示(前月リンク、当月、翌月リンク)
     */
    public function outCaption(Calendar $oCalendar)
    {
        // タイトル
        if (!$this->captionUse) {
            return;
        }

        $captionSide = $this->captionSide == 'bottom' ? ' style="caption-side:bottom"' : '';

        echo sprintf('<caption%s>', $captionSide);

        if ($this->captionLink) {
            $ppLink = $oCalendar->linkLast <= $oCalendar->prevPrev;
            $pLink = $oCalendar->linkLast <= $oCalendar->prevTime;
            $prevPrev = $this->_makeCaptionLink('prev_prev', $oCalendar->prevPrev, $ppLink);
            $prev = $this->_makeCaptionLink('prev', $oCalendar->prevTime, $pLink);
            echo sprintf('<div class="caption-prev">%s%s</div>', $prevPrev, $prev);
        }

        $title = $this->_date($this->month, $oCalendar->monthTime);

        echo sprintf('<div class="caption-title" data-month_time="%d">%s</div>', $oCalendar->monthTime, $title);

        if ($this->captionLink) {
            $nLink = $oCalendar->nextTime <= $oCalendar->linkFuture;
            $nnLink = $oCalendar->nextNext <= $oCalendar->linkFuture;
            $next = $this->_makeCaptionLink('next', $oCalendar->nextTime, $nLink);
            $nextNext = $this->_makeCaptionLink('next_next', $oCalendar->nextNext, $nnLink);
            echo sprintf('<div class="caption-next">%s%s</div>', $next, $nextNext);
        }

        echo "</caption>\n";
    }

    // キャプション内に表示する前月、翌月リンクのHTMLを生成する
    private function _makeCaptionLink($item, $datetime, $link=false)
    {
        if ($this->$item == '') {
            return '';
        }

        $mark = $this->_date($this->$item, $datetime);

        if ($link) {
            $linkUrl = add_query_arg(array('ct' => $datetime), $this->linkUrl);
            $linkUrl .= ($this->anchor == '' ? '' : "#{$this->anchor}");

            return $mark = sprintf('<a href="%s" rel="%d">%s</a>', $linkUrl, $datetime, $mark);
        }

        return sprintf('<span class="calendar-caption-link">%s</span>', $mark);
    }

    // カレンダー日付(TDセル)の表示
    private function _outDate($datetime)
    {
        if ($this->callTdCell) {
            call_user_func($this->callTdCell, $datetime);
            return;
        }

        $weekNo = (int) date('w', $datetime);
        $day = (int) date('j', $datetime);

        // td class
        $tdClass = array(
            sprintf('day-%d', $day),
            self::$weekClass[$weekNo],
        );

        if ($datetime == $this->oCalendar->todayTime) {
            $tdClass[] = 'today';
        }

        if ($this->callTdClass !== null) {
            $tdClass = call_user_func_array($this->callTdClass, array('datetime' => $datetime, 'classes' => $tdClass));
        }

        echo sprintf('<td class="%s">', implode(' ', $tdClass));

        // day number
        $dayOut = $this->callDayNumber === null ? '' : call_user_func($this->callDayNumber, $datetime);

        echo sprintf('<div class="day-number"><span class="day-number-str">%d</span>%s</div>', $day, $dayOut);

        // day content
        if ($this->callDayContent != null) {
            echo call_user_func($this->callDayContent, $datetime);
        }

        echo "</td>\n";
    }

    // カレンダー日付外(TDセル)の表示
    private function _outNoDate($weekNo)
    {
        if ($this->callTdNoDay) {
            call_user_func($this->callTdNoDay, $weekNo);
        } else {
            echo sprintf('<td class="no-day %s"></td>', self::$weekClass[$weekNo]) . "\n";
        }
    }

    /**
     * プロパティから読み出す
     */
    public function __get($key)
    {
        if (array_key_exists($key, $this->data)) {
            return $this->data[$key];
        }
/*
        if (isset($this->$key)) {
            return $this->$key;
        }
*/
        $trace = debug_backtrace();
        trigger_error(sprintf(
            "Undefined property: '%s&' in %s on line %d, E_USER_NOTICE",
            $key, $trace[0]['file'], $trace[0]['line']
        ));

        return null;
    }

    /**
     * プロパティをセットする
     */
    public function __set($key, $value)
    {
        if (array_key_exists($key, $this->data)) {
            $this->data[$key] = $value;
        //} else {
        //    $this->$key = $value;
        }

        return $value;
    }

}