<?php
/**
 * MTS Car Booking レンタル予定管理のビュー
 *
 * @Filename    BudgetAdminView.php
 * @Author      S.Hayashi
 * @Code        2018-08-16 Ver.1.0.0
 */
namespace MTSCarBookingTrial\views;

use MTSCarBookingTrial\Config;
use MTSCarBookingTrial\CarBooking;
use MTSCarBookingTrial\BudgetAdmin;
use MTSCarBookingTrial\models\Vehicle;
use MTSCarBookingTrial\models\Schedule;

class BudgetAdminView
{
    const PAGE_NAME = BudgetAdmin::PAGE_NAME;

    const JS = 'js/budget-admin.js';
    const CSS = 'css/mtsrcb-admin.css';
    const UI_CSS = 'https://code.jquery.com/ui/1.12.1/themes/smoothness/jquery-ui.css';
    //const CALENDAR_ID = 'mtsab-booking-calendar';

    private $oHelper;

    // ページURL
    private $pageUrl = '';
    private $listUrl = '';
    private $rentalUrl = '';

    // カレンダー操作オブジェクト
    private $oCalendar = null;

    // 営業カレンダー
    private $shopDays = array();

    // 車両スケジュール
    private $schedules = array();


    /**
     * Constructor
     */
    public function __construct()
    {
        // ページのURL
        $this->pageUrl = admin_url('admin.php?page=' . self::PAGE_NAME);

        // 予約リストのURL
        $this->listUrl = admin_url('admin.php?page=' . CarBooking::ADMIN_RENTAL_LIST);

        // 予約ページのURL
        $this->rentalUrl = admin_url('admin.php?page=' . CarBooking::ADMIN_RENTAL);

        $this->oHelper = FormHelper::getInstance();

        // JavaScript 組み込み登録の設定
        add_action('admin_enqueue_scripts', array($this, 'enqueueScripts'));
    }

    /**
     * JavaScript,Style headタグ内のロード登録
     */
    public function enqueueScripts()
    {
        $setting = Config::getSetting('reserve');

        // 営業時間
        $open = array(
            'open_hour' => (int) date('G', $setting->opening),
            'open_min' => (int) date('i', $setting->opening),
            'close_hour' => (int) date('G', $setting->closing),
            'close_min' => (int) date('i', $setting->closing),
        );

        $dir = dirname(__FILE__);

        wp_enqueue_script(self::PAGE_NAME, plugins_url(self::JS, $dir), array('jquery', 'jquery-ui-dialog'));
        wp_localize_script(self::PAGE_NAME, 'shop', $open);
        wp_enqueue_style(self::PAGE_NAME, plugins_url(self::CSS, $dir));
        wp_enqueue_style(self::PAGE_NAME . '-ui', self::UI_CSS);
    }

    /**
     * 営業カレンダー編集画面の表示
     */
    public function budgetView(BudgetAdmin $oBudgetAdmin)
    {
        $this->oCalendar = $oBudgetAdmin->oCalendar;
        $this->shopDays = $oBudgetAdmin->shopDays;
        $this->schedules = $oBudgetAdmin->schedules;

        // 登録車両一覧を取得する
        $vehicles = Vehicle::listVehicle();

?>
        <div class="wrap">
            <h1 class="wp-heading-inline">レンタル予定一覧</h1>
            <?php $this->_dispMessage($oBudgetAdmin->msgCode, $oBudgetAdmin->msgSub, $oBudgetAdmin->errflg) ?>

            <div class="action-row budget-action">
                <div class="budget-action-link">
                    <?php $this->_calendarLink() ?>
                </div>
                <div class="budget-action-rental">
                    予定表をクリックして車両のレンタル予約を実行します。<br>
                    <?php $this->_rentalLink($oBudgetAdmin->oSchedule, $vehicles) ?>
                </div>
            </div>

            <div id="budget-calendar">
                <?php foreach ($vehicles as $vehicleId => $vehicleTitle) {
                    $this->_budgetTable($vehicleId);
                } ?>
            </div>

            <div id="rental-time-dialog" style="display:none">
                <div id="rental-dialog-date">
                    <div id="rental-dialog-car"></div>
                    <label id="rental-dialog-title"></label>
                    <?php echo $this->oHelper->selectTime('rental-time', '', 'rental', 0); ?>
                </div>
            </div>
        </div>
<?php
        //ob_end_flush();
    }

    // カレンダー切替リンク
    private function _calendarLink()
    {
        $prev = add_query_arg(array('pt' => $this->oCalendar->prevTime), $this->pageUrl);
        $prevDisable = date('Y', $this->oCalendar->prevTime) < Config::RELEASE_YEAR ? ' disabled' : '';

        $next = add_query_arg(array('pt' => $this->oCalendar->nextTime), $this->pageUrl);
        $nextDisable = empty($this->shopDays) ? ' disabled' : '';

        echo sprintf('<a class="link-button button" href="%s"%s onclick="mtsrcb_budget_admin.turnCalendar(this)">＜</a>', $prev, $prevDisable);
        echo sprintf('<a class="link-button button" href="%s"%s onclick="mtsrcb_budget_admin.turnCalendar(this)">＞</a>', $next, $nextDisable);
        echo sprintf('<a class="link-button button" href="%s" onclick="mtsrcb_budget_admin.turnCalendar(this)">本日</a>', $this->pageUrl);
    }

    private function _rentalLink(Schedule $oSchedule, $vehicles)
    {
        $carTitle = $rentStart = $rentEnd = '';
        $vehicle_id = $rent_start = $rent_end = 0;

        if (0 < $oSchedule->vehicle_id) {
            $carTitle = $vehicles[$oSchedule->vehicle_id];
            $vehicle_id = $oSchedule->vehicle_id;

            if (0 < $oSchedule->rent_start) {
                $rentStart = date_i18n('Y年n月j日 (D) H:i', $oSchedule->rent_start);
                $rent_start = $oSchedule->rent_start;
            }

            if (0 < $oSchedule->rent_end) {
                $rentEnd = date_i18n('Y年n月j日 (D) H:i', $oSchedule->rent_end);
                $rent_end = $oSchedule->rent_end;
            }
        }

?>
        <form method="post" action="<?php echo $this->rentalUrl ?>">
            <input id="budget-car" class="budget-car-title" value="<?php echo $carTitle ?>" readonly>
            <input id="budget-start" class="budget-date-time input-focus" value="<?php echo $rentStart ?>" readonly>
            <input id="budget-end" class="budget-date-time" value="<?php echo $rentEnd ?>" readonly>
            <input id="budget-rental-do" type="submit" class="button" name="budget_rental" value="予約">
            <input id="budget-cancel-do" type="button" class="button" onclick="mtsrcb_budget_admin.formClear(); return false;" value="クリア">
            <input type="hidden" name="rental_nonce" value="<?php echo wp_create_nonce(CarBooking::ADMIN_RENTAL) ?>">
            <input id="budget-rent-start" type="hidden" name="rental[start]" value="<?php echo $rent_start ?>">
            <input id="budget-rent-end" type="hidden" name="rental[end]" value="<?php echo $rent_end ?>">
            <input id="budget-vehicle-id" type="hidden" name="rental[vehicle_id]" value="<?php echo $vehicle_id ?>">
        </form>
<?php
    }

    // 指定された車両のレンタル予定表
    private function _budgetTable($vehicleId)
    {
        $oVehicle = Vehicle::readVehicle($vehicleId);

        // 車両名称表示
        echo sprintf("<h3>%s</h3>\n", $oVehicle->oPost->post_title);
        echo sprintf('<table class="budget-table" data-vehicle_id="%d" data-vehicle_title="%s">',
            $vehicleId, $oVehicle->oPost->post_title) . "\n";

        // 予定表１行目 車両画像、年月カラムを表示する
        echo '<tr>';
        $this->_dispHeaderRow($oVehicle);
        echo "</tr>\n";

        // 予定表２行目 スケジュール日付を表示する
        echo '<tr>';
        $this->_dispDayRow();
        echo "</tr>\n";

        // 予定表３行目 レンタル予定を表示する
        echo '<tr>';
        $this->_dispRentalRow($vehicleId);
        echo "</tr>\n";

        echo "</table>\n";
    }

    // 予定表の１行目 先頭カラム画像と年月カラム表示
    private function _dispHeaderRow(Vehicle $oVehicle)
    {
        // 予定表に表示する車両の画像リンク または 名称を取り出す
        if (isset($oVehicle->image[0]) && $oVehicle->image[0] != '') {
            $car = sprintf('<img class="budget-title" src="%s">', $oVehicle->image[0]);
        } else {
            $car = sprintf('<span class="budget-title">%s</span>', $oVehicle->oPost->post_title);
        }

        // 予定表の先頭に車両画像(または名称)を表示する
        echo sprintf('<tr><th class="budget-name" rowspan="3">%s</th>', $car);

        // 月別に分割された年月を表示する
        foreach ($this->shopDays as $idxTime => $shopDays) {
            echo sprintf('<th class="budget-year-month" colspan="%d">%s</th>', count($shopDays), date_i18n('Y/n', $idxTime));
        }
        echo "</tr>\n";
    }

    // 予定表２行目 スケジュール日付を表示する
    private function _dispDayRow()
    {
        foreach ($this->shopDays as $idxTime => $shopDays) {
            foreach ($shopDays as $dayTime => $shopDay) {
                $class = $this->_dayClass($dayTime, $shopDay);
                echo sprintf('<td class="budget-day %s" data-daytime="%d">%s<br>%s</td>',
                    implode(' ', $class), $dayTime, date_i18n('j', $dayTime), date_i18n('D', $dayTime));
            }
        }
    }

    // 予定表３行目 レンタル予定を表示する
    private function _dispRentalRow($vehicleId)
    {
        foreach ($this->shopDays as $idxTime => $shopDays) {
            foreach ($shopDays as $dayTime => $shopDay) {
                $available = true;
                if (isset($this->schedules[$vehicleId])) {
                    $day = date('Ymd', $dayTime);
                    foreach ($this->schedules[$vehicleId] as $schedule) {
                        if (date('Ymd', $schedule['rent_start']) <= $day
                            && $day <= date('Ymd', $schedule['rent_end'])) {
                            $available = false;
                            break;
                        }
                    }
                }

                $class = $this->_dayClass($dayTime, $shopDay);

                $mark = '○';

                if (in_array('shop-close', $class)) {
                    $mark = 'ー';
                }

                if (!$available) {
                    $mark = sprintf('<a href="%s&amp;s_rental_id=%d">×</a>', $this->listUrl, $schedule['rental_id']);
                }

                $class = array($available ? 'budget-car' : 'rented-car') + $class;

                $memo = !empty($shopDay['memo']) ? "<br>{$shopDay['memo'][Config::LANG]}" : '';

                echo sprintf('<td class="%s" data-daytime="%d">%s%s</td>',
                    implode(' ', $class), $dayTime, $mark, $memo);
            }
        }
    }

    // 日付クラス設定
    private function _dayClass($dayTime, $shopDay)
    {
        $class = array();

        $class[] = strtolower(date('D', $dayTime));

        if ($dayTime == $this->oCalendar->todayTime) {
            $class[] = 'today';
        }

        if (!$shopDay['open']) {
            $class[] = 'shop-close';
        }

        if (!empty($shopDay['class'])) {
            $class[] = $shopDay['class'];
        }

        return $class;
    }

    // メッセージの表示
    private function _dispMessage($code, $sub='', $errflg=false)
    {
        static $message = array(
        );

        if (empty($code)) {
            return;
        }

        if (array_key_exists($code, $message)) {
            $msg = sprintf($message[$code], $sub);
        } else {
            $msg = sprintf("%s : %s", $code, $sub);
        }

        echo sprintf('<div class="%s"><p><strong>%s</strong></p></div>',
                ($errflg ? 'error' : 'updated'), $msg) . "\n";
    }

}
