<?php
/**
 * MTS Car Booking セッションデータ
 *
 * @Filename    Session.php
 * @Author      S.Hayashi
 * @Code        2018-09-03 Ver.1.0.0
 */
namespace MTSCarBookingTrial\models;

use MTSCarBookingTrial\Config;

class Session
{
    const MODULE = 'system';
    const MODEL = 'session';
    const VERSION = '1.0';

    // 予約登録セッションクッキー名称
    const SESSION_COOKIE = 'mtsrcb_session';

    const RETENTION = 7;

    // セッション格納ステータス
    public static $status = array(
        'new' => 1,
        'add' => 2,
        'edit' => 3,
    );

    // テーブル名
    private $sessionTable = '';

    // スケジュールデータ
    private $data = array(
        'session_id' => 0,
        'status' => 0,
        'cookie_id' => 0,
        'lang' => '',
        'meta' => array(),
        'updated' => '',
        'created' => '',
    );

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->sessionTable = Config::getTableName(self::MODEL);
    }

    /**
     * cookie_idをクッキーにセットする
     */
    public static function getCookieId()
    {
        // クッキーが設定済みならその値を戻す
        if (!empty($_COOKIE[self::SESSION_COOKIE])) {
            return $_COOKIE[self::SESSION_COOKIE];
        }

        // 重複しないcookieIdを生成する
        $cookieId = '';
        $oSession = null;
        for ($i = 0; $i < 5; $i++) {
            $cookieId = md5(uniqid(mt_rand(), true));
            $oSession = self::readSession($cookieId);
            if ($oSession->session_id <= 0) {
                setcookie(self::SESSION_COOKIE, $cookieId, 0, '/');
                break;
            }
        }

        return $cookieId;
    }

    /**
     * Cookie IDからSessionデータを取得する
     */
    public static function readSession($cookieId)
    {
        global $wpdb;

        $sql = $wpdb->prepare("SELECT * FROM " . Config::getTableName(self::MODEL) . " WHERE cookie_id=%s",
            $cookieId);

        $data = $wpdb->get_row($sql);

        $new = new Session;
        $new->cookie_id = $cookieId;

        if (!empty($data)) {
            $new->session_id = (int) $data->session_id;
            $new->status = $data->status;
            $new->lang = $data->lang;
            $new->meta = unserialize($data->meta);
            $new->updated = $data->updated;
            $new->created = $data->created;
        }

        return $new;
    }

    /**
     * セッション準備
     */
    public static function start($cookieId)
    {
        return self::readSession($cookieId);
    }

    /**
     * モデルクリア
     */
    public function clear($model)
    {
        if (isset($this->data['meta'][$model])) {
            unset($this->data['meta'][$model]);
            $this->_saveSession();
        }
    }

    /**
     * モデル取得
     */
    public function read($model)
    {
        if (isset($this->data['meta'][$model])) {
            return $this->data['meta'][$model];
        }

        return false;
    }

    /**
     * モデル保存
     */
    public function save($model='', $data='')
    {
        if (!empty($model)) {
            $this->data['meta'][$model] = $data;
        }

        return $this->_saveSession();
    }

    /**
     * Sessionデータを保存する
     */
    public function _saveSession()
    {
        // セッションデータを取得する
        //$oSession = self::readSession($cookieId);

        if ($this->session_id <= 0) {
            $ret = $this->_addSession();
        } else {
            $ret = $this->_updateSession();
        }

        return $ret;
    }

    // Sessionデータを追加する
    private function _addSession()
    {
        global $wpdb;

        //$this->data['meta'] = $meta;
        $this->created = $this->updated = current_time('mysql');

        $ret = $wpdb->insert(
            $this->sessionTable,
            array(
                'status' => $this->status,
                'cookie_id' => $this->cookie_id,
                'lang' => $this->lang,
                'meta' => serialize($this->data['meta']),
                'updated' => $this->updated,
                'created' => $this->created,
            ),
            array('%d', '%s', '%s', '%s', '%s', '%s')
        );

        if ($ret) {
            $ret = $wpdb->insert_id;
            $this->session_id = $ret;
        }

        return $ret;
    }

    /**
     * Sessionデータを更新する
     */
    private function _updateSession()
    {
        global $wpdb;

        //$this->data['meta'] = $meta;
        $this->updated = current_time('mysql');

        $ret = $wpdb->update(
            $this->sessionTable,
            array(
                'status' => $this->status,
                'lang' => $this->lang,
                'meta' => serialize($this->data['meta']),
                'updated' => $this->updated,
            ),
            array('session_id' => $this->session_id),
            array('%d', '%s', '%s', '%s'),
            array('%d')
        );

        if ($ret) {
            $ret = $this->session_id;
        }

        return $ret;
    }

    /**
     * Sessionデータをクリアする
     */
    public function clearSession()
    {
        // 古いセッションデータをテーブルから削除する
        $this->removeSession();

        // セッションデータを空にする
        return $this->_updateSession(array());
    }

    /**
     * 保存期日の過ぎたSessionデータを削除する
     */
    public function removeSession()
    {
        global $wpdb;

        $days = (int) abs(self::RETENTION);

        // 削除対象Unix Time
        $delTime = strtotime(date_i18n('Y-n-j')) - $days * 86400;

        $sql = $wpdb->prepare(
            "DELETE FROM {$this->sessionTable}"
            . " WHERE created<'%s'", date('Y-m-d H:i:s', $delTime));

        return $wpdb->query($sql);
    }

    /**
     * プロパティから読み出す
     */
    public function __get($key)
    {
        if (array_key_exists($key, $this->data)) {
            return $this->data[$key];
        }

        if (isset($this->$key)) {
            return $this->$key;
        }

        $trace = debug_backtrace();
        trigger_error(sprintf(
            "Undefined property: '%s&' in %s on line %d, E_USER_NOTICE",
            $key, $trace[0]['file'], $trace[0]['line']
        ));

        return null;
    }

    /**
     * プロパティをセットする
     */
    public function __set($key, $value)
    {
        if (array_key_exists($key, $this->data)) {
            $this->data[$key] = $value;
        } else {
            $this->$key = $value;
        }

        return $value;
    }

    /**
     * セッションデータテーブル生成
     */
    public static function installTable(&$version, $list)
    {
        // レンタルテーブル名を取得する
        $tableName = Config::getTableName(self::MODEL);

        // テーブルが登録されていない、またはバージョンが異なれば更新する
        if (array_key_exists(self::MODEL, $version) && in_array($tableName, $list)) {
            if ($version[self::MODEL] === self::VERSION) {
                return false;
            }
        }

        $sql = "CREATE TABLE $tableName (
            session_id int(11) unsigned NOT NULL AUTO_INCREMENT,
            status smallint(6) DEFAULT '0',
            cookie_id varchar(64) DEFAULT '',
            lang varchar(32) DEFAULT '',
            meta text,
            updated datetime DEFAULT NULL,
            created datetime DEFAULT NULL,
            PRIMARY KEY  (session_id),
            KEY cookie_id (cookie_id)
            ) DEFAULT CHARSET=utf8mb4;";

        dbDelta($sql);

        // バージョンデータの更新
        $version[self::MODEL] = self::VERSION;

        return true;
    }

}
