<?php
/**
 * MTS Car Booking メイン処理モジュール
 *
 * @Filename    CarBooking.php
 * @Author      S.Hayashi
 * @CODE        2018-06-27 Ver.1.0.0
 */
namespace MTSCarBookingTrial;

use MTSCarBookingTrial\models\Session;
use MTSCarBookingTrial\models\Schedule;
use MTSCarBookingTrial\models\Rental;
use MTSCarBookingTrial\models\Vehicle;
use MTSCarBookingTrial\models\Graphics;
use MTSCarBookingTrial\models\Mail;
use MTSCarBookingTrial\models\Alternate;

class CarBooking
{
    // オプションモジュール
    const MODULE = 'system';
    const MODEL = 'main';

    // メニュー表示位置
    const MENU_POSITION = 110;

    // 管理画面ページ カスタム投稿処理
    const ADMIN_VEHICLE = 'mtsrcb-vehicle';     // 車両
    const ADMIN_OPTION = 'mtsrcb-option';       // オプション
    const ADMIN_CHARGE = 'mtsrcb-charge';       // 料金表
    const ADMIN_MAIL_TEMPLATE = 'mtsrcb-mail';  // メールテンプレート

    // 管理画面ページ
    const ADMIN_TOP = 'mtsrcb-budget-trial';              // プラグインメニュートップ
    const ADMIN_BUDGET = 'mtsrcb-budget-trial';           // レンタル予定管理
    const ADMIN_RENTAL_LIST = 'mtsrcb-rental-list-trial'; // レンタル予約リスト管理
    const ADMIN_RENTAL = 'mtsrcb-rental-trial';           // 登録編集管理
    const ADMIN_PAYMENT = 'mtsrcb-payment-trial';         // レンタル料金管理
    const ADMIN_MAINTENANCE = 'mtsrcb-maintenance-trial'; // スケジュール管理
    const ADMIN_EMAIL = 'mtsrcb-email-trial';             // メール編集送信管理
    const ADMIN_CALENDAR = 'mtsrcb-calendar-trial';       // 管理画面営業カレンダー管理
    const ADMIN_SETTING = 'mtsrcb-setting-trial';         // 各種設定

    // フロントエンドページ(=スラッグ名)
    const BOOKING_PAGE = 'booking-car';             // 車両予約ページ
    const BOOKING_ENYTY_PAGE = 'booking-entry';     // 予約日付入力ページ
    const BOOKING_FORM_PAGE = 'booking-form';       // 予約フォーム入力ページ
    const BOOKING_ORDER_PAGE = 'booking-order';     // 予約実行ページ
    const BOOKING_ABORT_PAGE = 'booking-abort';     // 作業中止表示ページ
    const BOOKING_FINISH_PAGE = 'booking-finish';   // 予約終了ページ

    //const PAYMENT_PAGE = 'payment-car';             // 決済処理ページ

    // AJAX処理名
    const AJAX_ACTION = 'mtsrcb_ajax';

    // カスタムポストデータ定義モジュール
    public $oCustomPost = null;

    // 共通処理モジュール
    public $oConfig = null;                 // 環境設定

    // 管理画面カスタム投稿タイプ処理
    public $oVehicleAdmin = null;           // 車両管理
    public $oChargeAdmin = null;            // 料金表管理
    public $oOptionAdmin = null;            // オプション管理
    public $oMailTemplateAdmin = null;      // メールテンプレート管理

    // 管理処理オブジェクト
    public $oBudgetAdmin = null;            // レンタル予定管理
    public $oPaymentAdmin = null;           // レンタル料金処理
    public $oCalendarAdmin = null;          // 営業カレンダー管理
    public $oSettingAdmin = null;           // 各種設定

    // フロントエンド処理オブジェクト
    public $oBookingCar = null;             // 予約ページ

    // アクセス者情報
    public $cookieId = '';
    public $userId = 0;

    // セッションデータ
    public $oSession = null;

    // 対象フロントエンドページ
    private $frontPage = '';
    private $wrapPage = '';

    /**
     * Constructor
     */
    public function __construct()
    {
        add_action('init', array($this, 'initPlugin'));

        // カスタムポストデータ定義
        $this->oCustomPost = new CustomPost;
    }

    /**
     * プラグイン初期化処理
     */
    public function initPlugin()
    {
        // セッション処理用COOKIEを取得・設定する
        $this->cookieId = Session::getCookieId();

        // ログインユーザー情報の取得
        $this->userId = get_current_user_id();

        // 画像データ送信
        if (isset($_GET['mtsrcb_img']) && current_user_can(Config::WORKING_CAPABILITY)) {
            $this->_outImage(intval($_GET['mtsrcb_img']));
        }

        if (is_admin()) {
            // 管理画面メニュー登録
            add_action('admin_menu', array($this, 'adminMenu'));

            // 管理画面初期化処理
            add_action('admin_init', array($this, 'adminInit'));

        } else {
            // ショートコードの登録
            ShortCode::addShortCode();

            // フロントエンドヘッダー送信前予約処理
            add_action('wp', array($this, 'wpBooking'));

            // フロントエンド予約処理
            add_filter('the_content', array($this, 'contentBooking'), apply_filters('mtsrcb_content_priority', 11));
        }

        // 決済処理の終了処理フック
        add_filter('mts_close_payment', array($this, 'closePayment'), 15, 4);

        // AJAX ディスパッチャー登録
        add_action('wp_ajax_nopriv_' . self::AJAX_ACTION, array($this, 'ajaxAssist'));
        add_action('wp_ajax_' . self::AJAX_ACTION, array($this, 'ajaxAssist'));
    }

    // 画像データ送信
    private function _outImage($graphicsId)
    {
        Graphics::image($graphicsId);
    }

    /**
     * 管理画面 初期化処理
     */
    public function adminInit()
    {
        // 車両データ編集準備
        $this->oVehicleAdmin = new VehicleAdmin;

        // オプションデータの登録編集フック
        $this->oOptionAdmin = new OptionAdmin;

        // 料金表データ編集準備
        $this->oChargeAdmin = new ChargeAdmin;

        // メールテンプレート編集準備
        $this->oMailTemplateAdmin = new MailTemplateAdmin;

        // 各種設定 処理対象sectionの処理
        if ((isset($_GET['page']) && $_GET['page'] === self::ADMIN_SETTING)
         || (isset($_POST['option_page']) && $_POST['option_page'] === self::ADMIN_SETTING)) {
            $sections = Config::defaultSettings();
            $section = (isset($_REQUEST['section']) && in_array($_REQUEST['section'], $sections)) ? $_REQUEST['section'] : $sections[0];
            $this->oSettingAdmin = new SettingAdmin($section);
        }
    }

    /**
     * 管理画面メニューの登録
     */
    public function adminMenu()
    {
        // 管理画面サイドメニュー
        add_menu_page('レンタカー予約', 'レンタカー予約', Config::WORKING_CAPABILITY, self::ADMIN_TOP, array($this, 'adminBudget'), '', self::MENU_POSITION);

        $budgetHook = add_submenu_page(self::ADMIN_TOP, 'レンタル予定', 'レンタル予定', Config::WORKING_CAPABILITY, self::ADMIN_BUDGET, array($this, 'adminBudget'));
        add_action('load-' . $budgetHook, array($this, 'loadAdminBudget'));

        add_submenu_page(self::ADMIN_TOP, '予約リスト', '予約リスト', Config::WORKING_CAPABILITY, self::ADMIN_RENTAL_LIST, array($this, 'adminTrial'));

        add_submenu_page(self::ADMIN_TOP, 'レンタル登録編集', 'レンタル登録編集', Config::WORKING_CAPABILITY, self::ADMIN_RENTAL, array($this, 'adminTrial'));

        $paymentHook = add_submenu_page(self::ADMIN_TOP, 'レンタル料金', 'レンタル料金', Config::WORKING_CAPABILITY, self::ADMIN_PAYMENT, array($this, 'adminPayment'));
        add_action('load-' . $paymentHook, array($this, 'loadAdminPayment'));

        add_submenu_page(self::ADMIN_TOP, 'メンテナンス', 'メンテナンス', Config::WORKING_CAPABILITY, self::ADMIN_MAINTENANCE, array($this, 'adminTrial'));

        add_submenu_page(self::ADMIN_TOP, 'メール送信', 'メール送信', Config::WORKING_CAPABILITY, self::ADMIN_EMAIL, array($this, 'adminTrial'));

        $calendarHook = add_submenu_page(self::ADMIN_TOP, '営業カレンダー', '営業カレンダー', Config::WORKING_CAPABILITY, self::ADMIN_CALENDAR, array($this, 'adminCalendar'));
        add_action('load-' . $calendarHook, array($this, 'loadAdminCalendar'));

        $settingHook = add_submenu_page(self::ADMIN_TOP, '各種設定', '各種設定', Config::SETTING_CAPABILITY, self::ADMIN_SETTING, array($this, 'adminSetting'));
        add_action('load-' . $settingHook, array($this, 'initSettingView'));
    }

    /**
     * 各種設定初期化
     */
    public function initSettingView()
    {
        $this->oSettingAdmin->initSettingView();
    }

    /**
     * 各種設定処理
     */
    public function adminSetting()
    {
        $this->oSettingAdmin->setting();
    }

    /**
     * 管理画面　メニュー処理モジュールのロード
     *
     * レンタル予定管理ロード
     */
    public function loadAdminBudget()
    {
        $this->oBudgetAdmin = new BudgetAdmin;
    }

    // レンタル予定管理処理
    public function adminBudget()
    {
        $this->oBudgetAdmin->budget();
    }

    // レンタル料金管理ロード
    public function loadAdminPayment()
    {
        $this->oPaymentAdmin = new PaymentAdmin();
    }

    // レンタル料金管理
    public function adminPayment()
    {
        $this->oPaymentAdmin->payment();
    }

    // 営業カレンダー管理モジュールロード
    public function loadAdminCalendar()
    {
        $this->oCalendarAdmin = new CalendarAdmin;
    }

    // 営業カレンダー管理処理
    public function adminCalendar()
    {
        $this->oCalendarAdmin->shopCalendar();
    }

    // Trial版
    public function adminTrial()
    {
        echo '<h1>MTS Car Booking Trial</h1><p>このシステムは<a href="http://mtsrcb.mt-systems.jp">MTSレンタカー予約システム</a>の体験版です。</p>';
    }

    /**
     * ヘッダー送信前予約処理
     */
    public function wpBooking()
    {
        global $post;

        if (is_singular()) {
            $redirect = false;

            // 予約日付入力フォーム
            //if (preg_match('/^' . self::BOOKING_PAGE . '/', $post->post_name)) {
            if ($post->post_name === self::BOOKING_PAGE) {
                $this->frontPage = self::BOOKING_PAGE;
                $this->oBookingCar = new BookingCar;

                // レンタル予約処理を実行し表示ビューのスラッグ名を取得する
                $this->wrapPage = $this->oBookingCar->bookingEntry();
            }
        }
    }

    /**
     * フロントエンド予約処理
     */
    public function contentBooking($content)
    {
        $page = '';

        // 予約ページ
        if ($this->frontPage === self::BOOKING_PAGE) {
            switch ($this->wrapPage) {
                // 予約日付入力フォーム
                case self::BOOKING_ENYTY_PAGE:
                    $view = $this->oBookingCar->entryDate();
                    break;
                // 予約登録フォーム
                case self::BOOKING_FORM_PAGE:
                    $view = $this->oBookingCar->entryForm();
                    break;
                // 予約確認フォーム
                case self::BOOKING_ORDER_PAGE:
                    $view = $this->oBookingCar->orderForm();
                    break;
                // 予約中断フォーム
                case self::BOOKING_ABORT_PAGE:
                    $view = $this->oBookingCar->abortRental();
                    break;
                case self::BOOKING_FINISH_PAGE:
                    $view = $this->oBookingCar->finishBooking();
                    break;
            }

            // 出力ビューを多言語表示ページに差し込む
            $page = $this->_insertInPost($view);
        }

        // 出力ページに表示ページを挿入する
        if ($page) {
            $content = $this->_replaceContent($content, $page);
        }

        return $content;
    }

    // 多言語対応フロントページ差し込み処理
    private function _insertInPost($view)
    {
        $postName = sprintf('%s-%s', $this->wrapPage, $this->oSession->lang);

        $oPage = get_page_by_path($postName);

        if ($oPage) {
            $view = do_shortcode($this->_replaceContent($oPage->post_content, $view));
        }

        return $view;
    }

    /**
     * セッション処理開始とフロントエンドページ表示言語の設定
     */
    public function startSession()
    {
        $this->oSession = Session::start($this->cookieId);
        //$main = $this->oSession->read('main');

        $newLang = '';

        if ($this->oSession->lang == '') {
            $newLang = Config::LANG;
        }

        if (isset($_GET['lang']) && array_key_exists($_GET['lang'], Config::$languages)) {
            $newLang = $_GET['lang'];
        }

        if ($newLang && $this->oSession->lang != $newLang) {
            $this->oSession->lang = $newLang;
            $this->oSession->save();
        }

        return $this->oSession;
    }

    // コンテンツの挿入
    private function _replaceContent($content, $view)
    {
        if (preg_match('/%MTS_CONTENT%/', $content)) {
            $content = str_replace('%MTS_CONTENT%', $view, $content);
        } else {
            $content = $view . $content;
        }

        return $content;
    }

    /**
     * 決済処理へ移行する
     */
    public function redirectPayment(Rental $oRental, $mailVars)
    {
        return false;
    }

    /**
     * 決済処理の終了処理
     */
    public function closePayment($result, $rentalId, $applicationId='', $oJournal=null)
    {
        return false;
    }

    /**
     * 管理画面AJAX処理
     */
    public function ajaxAssist()
    {
        // NONCEチェック
        $nonce_key = isset($_POST['module']) ? $_POST['module'] : '';
        check_ajax_referer($nonce_key, 'nonce');

        $module = isset($_POST['module']) ? $_POST['module'] : '';
        $action = isset($_POST['method']) ? $_POST['method'] : '';

        $result = array('result' => false, 'html' => 'Failed.');

        // 処理振り分け
        switch ($module) {
            case self::ADMIN_RENTAL:
                if ($action == 'changeCalendar') {
                    $shopCalendar = RentalAdmin::shopCalendar((int) $_POST['mt'], (int) $_POST['vid']);
                    if ($shopCalendar) {
                        $result['result'] = true;
                        $result['html'] = $shopCalendar;
                    }
                }
                break;

            case self::BOOKING_PAGE:
                if ($action == 'changeCalendar') {
                    $post = stripslashes_deep($_POST);
                    $oBookingCar = new BookingCar;
                    $result['html'] = $oBookingCar->changeCalendar($post);
                    $result['result'] = !$oBookingCar->errflg;
                }
                break;
        }

        // 処理結果をJSONで戻す
        header("Content-Type: application/json; charset=utf-8");
        echo json_encode($result);

        exit();
    }

    /**
     * プラグイン組み込み
     */
    public function activate()
    {
        $update = false;

        // プラグインのバージョンを取得する
        $header = get_file_data(sprintf('%swp-content/plugins/%s/%s.php',
            ABSPATH, Config::PLUGIN_ID, Config::PLUGIN_ID), array('version' => 'Version'));

        // システム情報のインストールと取得
        $system = $this->_registry();

        // 各種設定データのインストールとアップデート
        if (SettingAdmin::installSetting()) {
            $update = true;
        }

        // データベーステーブルインストールのため処理モジュールを読み込む
        require_once(ABSPATH . "wp-admin/includes/upgrade.php");

        // スケジュールデータテーブルのインストールとアップデート
        if (Schedule::installTable($system['table'], $this->dbTableList())) {
            $update = true;
        };

        // レンタルテーブルのインストールとアップデート
        if (Rental::installTable($system['table'], $this->dbTableList())) {
            $update = true;
        };

        // セッションデータテーブルのインストール
        if (Session::installTable($system['table'], $this->dbTableList())) {
            $update = true;
        };

        // 画像データテーブルのインストール
        if (Graphics::installTable($system['table'], $this->dbTableList())) {
            $update = true;
        };

        // システム情報を更新する
        if ($update) {
            $system['version'] = $header['version'];
            update_option(Config::getOptionName(self::MODULE), $system);
        }
    }

    // データベーステーブルの一覧を取得する
    private function dbTableList()
    {
        global $wpdb;

        static $list = null;

        if (!is_null($list)) {
            return $list;
        }

        $tables = array();

        // データベースのテーブル一覧を取得する
        $data = $wpdb->get_results(sprintf("SHOW TABLES LIKE '%s%s%%'", $wpdb->prefix, Config::DOMAIN));

        foreach ($data as $key => $table) {
            foreach ($table as $tableName) {
                $tables[] = $tableName;
            }
        }

        $list = $tables;

        return $list;
    }

    // アプリケーションの登録(システム情報の登録)
    private function _registry()
    {
        // オプションデータのシステム情報を取得する
        $optionName = Config::getOptionName(self::MODULE);
        $system = get_option($optionName, array());

        // システム情報の新規登録
        if (empty($system)) {
            $system = Config::$system;
            $system['application'] = Config::APPLICATION_ID;

            add_option($optionName, $system, '', 'no');
        }

        return $system;
    }

    /**
     * プラグイン削除
     */
    public function uninstall()
    {
    }

}
