<?php
/**
 * MTS Car Booking レンタル料金管理のビュー
 *
 * @Filename    PaymentAdminView.php
 * @Author      S.Hayashi
 * @Code        2018-11-19 Ver.1.0.0
 */
namespace MTSCarBookingTrial\views;

use MTSCarBookingTrial\Config;
use MTSCarBookingTrial\PaymentAdmin;
use MTSCarBookingTrial\models\Charge;
use MTSCarBookingTrial\models\Rental;
use MTSCarBookingTrial\models\Vehicle;
use MTSCarBookingTrial\models\Customer;
use MTSCarBookingTrial\lang\Translation;

class PaymentAdminView
{
    const PAGE_NAME = PaymentAdmin::PAGE_NAME;

    const JS = 'js/payment-admin.js';
    const CSS = 'css/mtsrcb-admin.css';

    private $oHelper;

    private $lang = Config::LANG;
    private $nonce = '';

    // レンタル料金モジュール
    private $oPaymentAdmin = null;

    /**
     * Constructor
     */
    public function __construct(PaymentAdmin $oPaymentAdmin)
    {
        $this->oPaymentAdmin = $oPaymentAdmin;

        $this->nonce = wp_create_nonce(self::PAGE_NAME);

        $this->oHelper = FormHelper::getInstance();

        // JavaScript 組み込み登録の設定
        add_action('admin_enqueue_scripts', array($this, 'enqueueScripts'));
        add_action("admin_print_scripts", array($this, 'enqueueData'));
    }

    /**
     * JavaScript,Style headタグ内のロード登録
     */
    public function enqueueScripts()
    {
        $dir = dirname(__FILE__);

        wp_enqueue_script(self::PAGE_NAME, plugins_url(self::JS, $dir), array('jquery'));
        wp_enqueue_style(self::PAGE_NAME, plugins_url(self::CSS, $dir));
    }

    /**
     * JavaScript定義データ挿入
     */
    public function enqueueData()
    {
?>
        <script type='text/javascript'>
            let mtsrcb_currency=<?php echo sprintf("'%s'", $this->oPaymentAdmin->setting['payment']->currency) ?>;
            let mtsrcb_decimals=<?php echo sprintf("%d", Config::$currency[$this->oPaymentAdmin->setting['payment']->currency]) ?>;
        </script>
<?php
    }

    // ページタイトル表示
    private function _dispPageTitle()
    {
?>
        <h1 class="wp-heading-inline">レンタル料金</h1>
<?php
    }

    // メッセージの表示
    private function _dispMessage()
    {
        static $message = array(
            'NOT_FOUND' => 'レンタルデータは見つかりませんでした (予約ID %s)',

            'SAVED_NORMALLY' => '正常に保存しました',
        );

        if (empty($this->oPaymentAdmin->msgCode)) {
            return;
        }

        if (array_key_exists($this->oPaymentAdmin->msgCode, $message)) {
            $msg = sprintf($message[$this->oPaymentAdmin->msgCode], $this->oPaymentAdmin->msgSub);
        } else {
            $msg = sprintf("%s : %s", $this->oPaymentAdmin->msgCode, esc_html($this->oPaymentAdmin->msgSub));
        }

        echo sprintf('<div class="%s"><p><strong>%s</strong></p></div>',
                ($this->oPaymentAdmin->errflg ? 'error' : 'updated'), $msg) . "\n";
    }

    /**
     * レンタルデータ検索画面の表示
     */
    public function findView()
    {

?>
        <div class="wrap">
            <?php $this->_dispPageTitle();
            $this->_dispMessage(); ?>

            <h3>予約リスト検索</h3>
            <form method="post">
                <div class="mts-v-middle">
                    <label for="reserve-rental-id">予約ID：</label>
                    <input id="reserve-rental-id" type="text" name="rental[reserve_id]" value="<?php echo esc_html($this->oPaymentAdmin->oRental->reserve_id) ?>">
                    <input type="submit" class="button" name="find_rental" value="検索する">
                </div>
                <input type="hidden" name="payment_nonce" value="<?php echo $this->nonce ?>">
            </form>
        </div>

<?php
    }


    /**
     * 料金編集画面の表示
     */
    public function paymentView()
    {
        $oRental = $this->oPaymentAdmin->oRental;
        $oVehicle = $this->oPaymentAdmin->oVehicle;

?>
        <div class="wrap">
            <?php $this->_dispPageTitle();
            $this->_dispMessage(); ?>

            <div class="rental-form-block">
                <h3 class="payment-title">レンタル内容</h3>
                <form method="post">
                    <?php $this->_dispRentalPayment($oRental, $oVehicle) ?>
                    <div class="payment-action">
                        <input type="submit" class="button-primary" name="update_accounting" value="保存する">
                        <input type="hidden" name="rental[rental_id]" value="<?php echo $oRental->rental_id ?>">
                        <input type="hidden" name="payment_nonce" value="<?php echo $this->nonce ?>">
                    </div>
                </form>
            </div>

            <div class="rental-form-block">
                <h3 class="payment-title">料金計算</h3>
                <form method="post">
                    <?php $this->_chargeForm($oRental, $oVehicle) ?>
                    <div class="payment-action">
                        <input type="submit" class="button-primary" name="save_charge" value="保存する">
                        <input type="submit" class="button" name="input_charge" value="入力計算">
                        <input type="submit" class="button" name="init_charge" value="初期計算">
                        <input type="hidden" name="rental[rental_id]" value="<?php echo $oRental->rental_id ?>">
                        <input type="hidden" name="payment_nonce" value="<?php echo $this->nonce ?>">
                    </div>
                </form>
            </div>

            <div class="rental-form-block">
                <h3 class="payment-title">燃料代計算</h3>
                <form method="post">
                    <?php $this->_fuelForm($oRental, $oVehicle) ?>
                    <div class="payment-action">
                        <input type="submit" class="button-primary" name="save_fuel" value="保存する">
                        <input id="fuel-cost-compute" type="button" class="button" value="計算する">
                        <input type="hidden" name="rental[rental_id]" value="<?php echo $oRental->rental_id ?>">
                        <input type="hidden" name="payment_nonce" value="<?php echo $this->nonce ?>">
                    </div>
                </form>
            </div>

            <div class="rental-form-block">
                <h3 class="payment-title">キャンセル料金計算</h3>
                <form method="post">
                    <?php $this->_cancelForm($oRental, $oVehicle) ?>
                </form>
            </div>

            <div class="payment-block">
                <h3 class="payment-title">申込者(利用者)</h3>
                <?php $this->_dispCustomer($oRental->oCustomer) ?>
            </div>

        </div>

<?php
    }

    // 料金計算フォーム
    private function _chargeForm(Rental $oRental, Vehicle $oVehicle)
    {
        // 料金設定
        $setting = $this->oPaymentAdmin->setting['payment'];

        // 料金の保存データを取得する
        $payment = $oRental->getPayment('charge');

        // 通貨
        $currency = isset($payment['currency']) ? $payment['currency'] : $setting->currency;

        // 通貨の小数点以下桁数
        $decimals = isset(Config::$currency[$currency]) ? Config::$currency[$currency] : 0;

        // 合計額
        $total = isset($payment['total']) ? $payment['total'] : 0;

        // 消費税率
        $taxRate = isset($payment['taxRate']) ? $payment['taxRate'] : $setting->tax;

        // 消費税額
        $tax = isset($payment['tax']) ? $payment['tax'] : 0;

        // 消費税方式
        $consumption = isset($payment['consumption']) ?
            ParamAdmin::$consumption[$payment['consumption']] : ParamAdmin::$consumption[$setting->consumption];

?>
        <table class="mts-view-table">
            <tr>
                <th></th><td>料金表</td><td>計算方法</td><td class="price-col title">料金(<?php echo ParamAdmin::$currency[$currency] ?>)</td>
            </tr>
            <tr>
                <th><?php echo $oVehicle->oPost->post_title ?></th>
                <?php $this->_dispChargeMethod($oRental->oVehicleSchedule->schedule_id, (float) $oRental->oVehicleSchedule->charge, $oVehicle->oCharge, $decimals) ?>
            </tr>
            <?php foreach ($oRental->optionSchedules as $oSchedule) :
                if ($oSchedule->select) :
                    $oOption = $oVehicle->options[$oSchedule->option_id]; ?><tr>
                <th><?php echo $oOption->oPost->post_title ?></th>
                <?php $this->_dispChargeMethod($oSchedule->schedule_id, (float) $oSchedule->charge, $oOption->oCharge, $decimals) ?>
            </tr><?php endif; endforeach; ?>
            <tr>
                <th rowspan="3">合計計算</th>
                <td class="total-title total" colspan="2">合計</td>
                <td class="price-col"><?php
                    echo sprintf('<input type="text" id="total-price" class="charge-cost" value="%s" readonly>',
                        number_format($total, $decimals)) ?></td>
            </tr>
            <tr>
                <td class="total-title tax" colspan="2"><?php echo sprintf('消費税%d%%(%s)', $taxRate, $consumption) ?></td>
                <td class="price-col"><?php
                    echo sprintf('<input type="text" id="tax-price" class="charge-cost" value="%s" readonly>', number_format($tax, $decimals)) ?></td>
            </tr>
            <tr>
                <td class="total-title grand-total" colspan="2">総合計</td>
                <td class="price-col"><?php
                    echo sprintf('<input type="text" id="grand-total-price" class="charge-cost" name="rental[charge_total]" value="%s" readonly>',
                        number_format($oRental->charge_total, $decimals))?></td>
            </tr>
        </table>

<?php
    }

    // 料金表名と計算方法を出力する
    private function _dispChargeMethod($scheduleId, $val, $oCharge, $decimals)
    {
        if (empty($oCharge)) {
            echo '<td colspan="3">料金設定が未定です</td>';
            return;
        }
?>
        <td><?php echo $oCharge->oPost->post_title ?></td>
        <td><?php echo ParamAdmin::$compMethod[$oCharge->charge->method] ?></td>
        <td class="price-col"><?php echo sprintf('<input type="text" id="schedule-%d" class="charge-cost" name="payment[%d]" value="%s">',
                $scheduleId, $scheduleId, number_format($val, $decimals)) ?></td>

<?php
    }

    // 燃料代計算フォーム
    private function _fuelForm(Rental $oRental, Vehicle $oVehicle)
    {
        // 料金設定
        $setting = $this->oPaymentAdmin->setting['payment'];

        // 通貨の小数点以下桁数
        $decimals = isset(Config::$currency[$setting->currency]) ? Config::$currency[$setting->currency] : 0;


        // 燃料代
        $fuelCharge = isset($oRental->payment['fuel']) ? $oRental->payment['fuel'] : 0;

        // 燃料代単価
        $fuelMileage = isset($oRental->payment['mileage']) ? $oRental->payment['mileage'] : 0;

?>
        <table class="mts-view-table">
            <tr>
                <th></th><td>走行距離と単価</td><td class="price-col title">料金(<?php echo ParamAdmin::$currency[$setting->currency] ?>)</td>
            </tr>
            <tr>
                <th>燃料代</th>
                <td>
                    <div class="fuel-cost">
                        <label class="fuel-km">出発</label><input type="text" id="odo-start" class="odo-km" name="schedule[odo_start]" value="<?php
                            echo $oRental->oVehicleSchedule->odo_start ?>">
                    </div>
                    <div class="fuel-cost">
                        <label class="fuel-km">返却</label><input type="text" id="odo-end" class="odo-km" name="schedule[odo_end]" value="<?php
                            echo $oRental->oVehicleSchedule->odo_end ?>">
                    </div>
                    <div class="fuel-cost">
                        <label class="fuel-km">1L単価</label><input type="text" class="charge-cost" id="fuel-cost" name="payment[mileage]" value="<?php echo number_format($fuelMileage, $decimals) ?>">
                        <input type="hidden" id="fuel-mileage" value="<?php echo $oVehicle->charge->fuel_km ?>">
                    </div>
                </td>
                <td class="price-col fuel-charge">
                    <input type="text" id="fuel-charge" class="charge-cost" name="payment[fuel]" value="<?php echo number_format($fuelCharge, $decimals) ?>">
                </td>
            </tr>
        </table>

<?php
    }

    // キャンセル料金計算フォーム
    private function _cancelForm(Rental $oRental, Vehicle $oVehicle)
    {
        // 料金設定
        $settingCancel = Config::getSetting('cancel');
        $settingPayment = Config::getSetting('payment');

        // キャンセル料金を請求しない
        if ($settingCancel->charge <= 0) {
            echo '<p>キャンセル料は請求しない設定です。</p>';
            return;
        }

        // 料金の保存データを取得する
        $payment = $oRental->getPayment('cancel');

        // 通貨
        $currency = isset($payment['currency']) ? $payment['currency'] : $settingPayment->currency;

        // 通貨の小数点以下桁数
        $decimals = isset(Config::$currency[$currency]) ? Config::$currency[$currency] : 0;

        // キャンセル料計算方式
        $chargeSetting = ParamAdmin::$cancelChargeType[$settingCancel->charge];

        // 合計額
        $cancelCharge = isset($payment['cancelCharge']) ? $payment['cancelCharge'] : 0;

        // 消費税率
        $taxRate = isset($payment['taxRate']) ? $payment['taxRate'] : $settingPayment->tax;

        // 消費税額
        $tax = isset($payment['tax']) ? $payment['tax'] : 0;

        // 消費税方式
        $consumption = isset($payment['consumption']) ?
            ParamAdmin::$consumption[$payment['consumption']] : ParamAdmin::$consumption[$settingPayment->consumption];

        // キャンセル料合計
        $cancelTotal = isset($payment['cancelTotal']) ? $payment['cancelTotal'] : 0;

        // キャンセル情報
        $cancelInfo = '';
        $cancelTime = isset($payment['cancelTime']) ? $payment['cancelTime'] : 0;

        if (0 < $cancelTime) {
            $cancelAvailable = $oRental->cancelAvailable($payment['cancelTime']);
            if ($cancelAvailable) {
                $cancelInfo = sprintf('<p>有効期限：%s まで,　キャンセル日：%s</p>',
                    date('Y-n-j', $cancelAvailable['limit']), date('Y-n-j', $payment['cancelTime'])) . "\n";
            }
        }

        echo $cancelInfo;
?>
        <table class="mts-view-table">
            <tr>
                <th class="total-title"></th><td "charge-title">計算方法</td><td class="price-col title">料金</td>
            </tr>
            <tr>
                <th class="reserve-title">キャンセル料</th>
                <td><?php echo $chargeSetting ?></td>
                <td class="price-col">
                    <input type="text" class="charge-cost cancel-charge" name="cancel[charge]" value="<?php echo number_format($cancelCharge, $decimals) ?>">
                </td>
            </tr>
            <tr>
                <td class="total-title cancel-tax" colspan="2"><?php echo sprintf('消費税%d%%(%s)', $taxRate, $consumption) ?></td>
                <td class="price-col cancel-tax">
                    <input type="text" class="charge-cost cancel-charge" name="cancel[tax]" value="<?php echo number_format($tax, $decimals) ?>" readonly>
                </td>
            </tr>
            <tr>
                <td class="total-title cancel-total" colspan="2">合計</td>
                <td class="price-col cancel-total">
                    <input type="text" class="charge-cost cancel-charge" name="cancel[total]" value="<?php echo number_format($cancelTotal, $decimals) ?>" readonly>
                </td>
            </tr>
        </table>

        <div class="payment-action">
            <input type="submit" class="button-primary" name="save_cancel" value="保存する">
            <input type="submit" class="button" name="input_cancel" value="入力値">
            <input type="submit" class="button" name="init_cancel" value="現在時刻">
            <input type="hidden" name="rental[rental_id]" value="<?php echo $oRental->rental_id ?>">
            <input type="hidden" name="payment_nonce" value="<?php echo $this->nonce ?>">
        </div>

<?php
    }

    // レンタル予約データを表示する
    private function _dispRentalPayment(Rental $oRental, Vehicle $oVehicle)
    {
?>
        <table class="mts-view-table">
            <tr>
                <th>予約ID(レンタルID)</th>
                <td><?php echo sprintf('%s (%d)', $oRental->reserve_id, $oRental->rental_id) ?></td>
            </tr>
            <tr>
                <th>利用期間</th>
                <td>
                    <div class="rental-period-date"><?php $this->_dispDateTime($oRental->rent_start) ?></div>
                    <div class="rental-period-arrow"><?php echo Translation::$pArrow[$this->lang] ?></div>
                    <div class="rental-period-date"><?php $this->_dispDateTime($oRental->rent_end) ?></div>
                </td>
            </tr>
            <tr>
                <th>車両</th>
                <td><?php if (0 < $oVehicle->postId) {
                    echo sprintf('<span class="payment-car">%s%s %s %s</span><span class="payment-car">%s</span>',
                        $oVehicle->number->region, $oVehicle->number->class, $oVehicle->number->kana, $oVehicle->number->designation, $oVehicle->oPost->post_title);
                    echo sprintf('<span class="payment-car">%s</span><span class="payment-car">燃費%skm/L</span>',
                        ParamAdmin::$fuelType[$oVehicle->charge->fuel], $oVehicle->charge->fuel_km);
                } ?></td>
            </tr>
            <tr class="accounting-row">
                <th>決済状況</th>
                <td>
                    <?php echo $this->oHelper->radioButton('pay-accounting', 'accounting-input', "rental[accounting]", ParamAdmin::$accounting, $oRental->accounting) ?>
                </td>
            </tr>
        </table>

<?php
    }

    // 予約日時の表示
    private function _dispDateTime($datetime)
    {
        if ($this->lang == 'ja') {
            $date = date_i18n(Translation::$dateFormat[$this->lang], $datetime);
        } else {
            $date = date(Translation::$dateFormat[$this->lang], $datetime);
        }

        $time = date(Translation::$timeFormat[$this->lang], $datetime);

        echo sprintf('<span class="rental-period-day">%s</span>', $date);
        echo sprintf('<span class="rental-period-time">%s</span>', $time);
    }







    // 予約者情報表示
    private function _dispCustomer(Customer $oCustomer)
    {
        $oCustomerView = new CustomerView;

        $setting = $this->oPaymentAdmin->setting['customer'];

        $oCustomerView->confirmationView($oCustomer, (array) $setting);
    }

}
