<?php
/**
 * MTS フォーム出力ヘルパー処理モジュール
 *
 * @Filename    FormHelper.php
 * @Author      S.Hayashi
 * @Code        2018-07-13 MTSCarBookingTrial Ver.1.0.0
 */
namespace MTSCarBookingTrial\views;

use MTSCarBookingTrial\Config;

class FormHelper
{
    // Singleton
    private static $instance = NULL;

    public static $week = array(
        'sun' => '日', 'mon' => '月', 'tue' => '火', 'wed' => '水', 'thu' => '木', 'fri' => '金', 'sat' => '土'
    );

    public static $donot = array(
        0 => 'しない',
        1 => 'する',
    );

    public static $reki = array(
        'seireki' => '西暦',
        'meiji' => '明治',
        'taisho' => '大正',
        'showa' => '昭和',
        'heisei' => '平成',
        'reiwa' => '令和',
    );

    public static $nengo = array(
        'seireki' => '00010101',
        'meiji' => '18680125',
        'taisho' => '19120730',
        'showa' => '19261225',
        'heisei' => '19890108',
        'reiwa' => '20190501',
    );

    /**
     * インスタンス化
     */
    public static function getInstance()
    {
        if (empty(self::$instance)) {
            self::$instance = new FormHelper;
        }

        return self::$instance;
    }

    /**
     * 都道府県入力セレクトボックス
     */
    public function selectPref($id, $class, $name, $val)
    {
        static $prefs = array('', '北海道', '青森県', '岩手県', '宮城県', '秋田県', '山形県', '福島県', '茨城県', '栃木県',
            '群馬県', '埼玉県', '千葉県', '東京都', '神奈川県', '新潟県', '富山県', '石川県', '福井県', '山梨県', '長野県',
            '岐阜県', '静岡県', '愛知県', '三重県', '滋賀県', '京都府', '大阪府', '兵庫県', '奈良県', '和歌山県', '鳥取県',
            '島根県', '岡山県', '広島県', '山口県', '徳島県', '香川県', '愛媛県', '高知県', '福岡県', '佐賀県', '長崎県',
            '熊本県', '大分県', '宮崎県', '鹿児島県', '沖縄県');

        ob_start();
        echo sprintf('<select id="%s" class="%s" name="%s">', $id, $class, $name) . "\n";

        foreach ($prefs as $pref) {
            $title = empty($pref) ? '----' : $pref;
            echo sprintf('<option value="%s"%s>%s</option>',
                    $pref, $this->cmpSetSelected($pref, $val), $title) . "\n";
        }

        echo "</select>\n";
        return ob_get_clean();
    }

    /**
     * 性別ラジオボタン入力
     */
    public function checkGender($id, $class, $name, $val)
    {
        ob_start();

        echo sprintf('<input id="%s-male" class="%s" type="radio" name="%s" value="男性"%s>', $id, $class, $name, $this->setChecked('男性', $val));
        echo sprintf('<label for="%s-male" class="%s-label">男性</label>', $id, $class);
        echo sprintf('<input id="%s-female" class="%s" type="radio" name="%s" value="女性"%s>', $id, $class, $name, $this->setChecked('女性', $val));
        echo sprintf('<label for="%s-female" class="%s-label">女性</label>', $id, $class);

        return ob_get_clean();
    }

    private function _getId($value)
    {
        return empty($value) ? '' : sprintf(' id="%s"', $value);
    }

    private function _getClass($value)
    {
        return empty($value) ? '' : sprintf(' class="%s"', $value);
    }

    private function _getName($value, $add='')
    {
        if ($value == '' && $add == '') {
            return '';
        }

        return empty($value) ? sprintf(' name="%s"', $add) : sprintf(' name="%s[%s]"', $value, $add);
    }

    /**
     * 西暦・元号選択
     */
    public function selectGengo($id, $class, $name, $gengo='', $value='1-1-1')
    {
        if (empty($gengo)) {
            $gengo = $this->getGengo($value);
        }

        return $this->selectBox($id, $class, $name, self::$reki, $gengo);
    }

    /**
     * 元号を戻す
     */
    public function getGengo($value='1-1-1')
    {
        $gengo = 'seireki';

        $ymd = explode('-', $value);

        if (count($ymd) === 3) {
            $date = sprintf('%04d%02d%02d', $ymd[0], $ymd[1], $ymd[2]);

            foreach (self::$nengo as $key => $start) {
                if ($date < $start) {
                    break;
                }
                $gengo = $key;
            }
        }

        return $gengo;
    }

    /**
     * 年号を戻す
     */
    public function getNengo($value='0001-01-01', $gengo='seireki')
    {
        $nengo = intval(substr($value, 0, 4));

        if (array_key_exists($gengo, self::$nengo) && $gengo !== 'seireki') {
            $gannen = intval(substr(self::$nengo[$gengo], 0, 4));
            var_dump($gannen);
            $nengo = $nengo - $gannen + 1;
        }

        return $nengo;
    }

    // 年セレクトボックス
    public function selectYear($id, $class, $name, $val, $startYear, $endYear, $suffix='年')
    {
        $increment = $startYear < $endYear ? 1 : -1;

        ob_start();
        echo sprintf('<select %s%s%s>', $this->_getId($id), $this->_getClass($class), $this->_getName($name, 'year'));
        echo '<option value="">----</option>';
        for ($i = $startYear; $i != $endYear; $i += $increment) {
            echo sprintf('<option value="%d"%s>%d</option>', $i, $this->cmpSetSelected($val, $i), $i);
        }
        echo sprintf('<option value="%d"%s>%d</option>', $i, $this->cmpSetSelected($val, $i), $i);
        echo sprintf("</select>%s\n", $this->_suffix($suffix));

        return ob_get_clean();
    }

    // 月セレクトボックス
    public function selectMonth($id, $class, $name, $val, $suffix='月')
    {
        ob_start();
        echo sprintf('<select %s%s%s>', $this->_getId($id), $this->_getClass($class), $this->_getName($name, 'month'));
        echo '<option value="">--</option>';
        for ($i = 1; $i <= 12; $i++) {
            echo sprintf('<option value="%d"%s>%d</option>', $i, $this->cmpSetSelected($val, $i), $i);
        }
        echo sprintf("</select>%s\n", $this->_suffix($suffix));
        return ob_get_clean();
    }

    // 日セレクトボックス
    public function selectDay($id, $class, $name, $val, $suffix='日')
    {
        ob_start();
        echo sprintf('<select %s%s%s>', $this->_getId($id), $this->_getClass($class), $this->_getName($name, 'day'));
        echo '<option value="">--</option>';
        for ($i = 1; $i <= 31; $i++) {
            echo sprintf('<option value="%d"%s>%d</option>', $i, $this->cmpSetSelected($val, $i), $i);
        }
        echo sprintf("</select>%s\n", $this->_suffix($suffix));
        return ob_get_clean();
    }

    // 曜日セレクトボックス
    public function selectWeek($id, $class, $name, $val, $suffix='曜日')
    {
        $pId = empty($id) ? '' : sprintf(' id="%s"', $id);
        $pClass = empty($class) ? '' : sprintf(' class="%s"', $class);
        $pName = empty($name) ? '' : sprintf(' name="%s"', $name);

        ob_start();
        echo sprintf('<select%s%s%s>', $pId, $pClass, $pName);
        echo '<option value="-1">--</option>';
        $i = 0;
        foreach (self::$week as $key => $title) {
            echo sprintf('<option value="%s"%s>%s</option>',
                $key, $this->cmpSetSelected($val, $i), $title);
            $i++;
        }
        echo sprintf("</select>%s\n", $this->_suffix($suffix));
        return ob_get_clean();
    }

    /**
     * 誕生日入力セレクトボックス
     */
    public function selectBirthday($id, $class, $name, $val='')
    {
        $year = empty($val) ? 0 : intval(current(explode('-', $val)));

        $startYear = intval(date_i18n('Y'));
        $endYear = $startYear - 100;
        if ($year !== 0 && $year < $endYear) {
            $endYear = $year;
        }

        return $this->selectDate($id, $class, $name, $val, $startYear, $endYear);
    }

    /**
     * 年月日入力セレクトボックス
     */
    public function selectDate($id, $class, $name, $val='', $startYear=0, $endYear=0)
    {
        // 初期値
        $date = empty($val) ? '0-0-0' : $val;

        $valdate = explode('-', $date);
        $start = intval($startYear);
        $end = intval($endYear);

        // 年
        $dateForm = $this->selectYear("{$id}-year", $class, $name, intval($valdate[0]), $start, $end);

        // 月
        $dateForm .= $this->selectMonth("{$id}-month", $class, $name, intval($valdate[1]));

        // 日
        $dateForm .= $this->selectDay("{$id}-day", $class, $name, intval($valdate[2]));

        return $dateForm;
    }

    /**
     * 時間入力セレクトボックス
     */
    public function selectTime($id, $class, $name, $val=10 * 3600, $param=array('start' => 0, 'end' => 24, 'interval' => 5))
    {
        $classStr = empty($class) ? '' : sprintf(' class="%s"', $class);
        $val %= 86400;
        $hVal = (int) ($val / 3600);
        $mVal = (int) ($val % 3600 / 60);

        ob_start();

        // 時
        $idStr = empty($id) ? '' : sprintf(' id="%s-hour"', $id);
        $nameStr = empty($name) ? '' : sprintf(' name=%s[hour]', $name);
        echo sprintf('<select%s%s%s>', $idStr, $classStr, $nameStr) . "\n";
        for ($i = $param['start']; $i <= $param['end']; $i++) {
            //$h = $i * 3600;
            echo sprintf('<option value="%d"%s>%02d</option>', $i, $this->cmpSetSelected($i, $hVal), $i) . "\n";
        }
        echo '</select><span class="form-addchr">:</span>';

        // 分
        $idStr = empty($id) ? '' : sprintf(' id="%s-minute"', $id);
        $nameStr = empty($name) ? '': sprintf(' name=%s[minute]', $name);
        echo sprintf('<select%s%s%s>', $idStr, $classStr, $nameStr) . "\n";
        for ($i = 0; $i < 60; $i += $param['interval']) {
            //$m = $i * 60;
            echo sprintf('<option value="%d"%s>%02d</option>', $i, $this->cmpSetSelected($i, $mVal), $i) . "\n";
        }
        echo "</select>\n";

        return ob_get_clean();
    }

    // 日時のセレクトボックスデフォルト選択設定
    public function cmpSetSelected($val1, $val2)
    {
        return $val1 === $val2 ? ' selected="selected"' : '';
    }

    /**
     * 一般的なセレクトボックス
     */
    public function selectBox($id, $class, $name, $options, $value, $suffix='')
    {
        $val = explode(',', $value);
        $pId = empty($id) ? '' : sprintf(' id="%s"', $id);
        $pClass = empty($class) ? '' : sprintf(' class="%s"', $class);
        $pName = empty($name) ? '' : sprintf(' name="%s"', $name);

        ob_start();

        echo sprintf('<select%s%s%s>', $pId, $pClass, $pName);

        foreach ($options as $idx => $item) {
            $title = $item === '' ? '----' : $item;
            echo sprintf('<option value="%s"%s>%s</option>', $idx, $this->setSelected($idx, $val), $title);
        }

        echo sprintf("</select>%s\n", $suffix ? $this->_suffix($suffix) : '');

        return ob_get_clean();
    }

    /**
     * セレクトボックス用ナンバー選択配列生成
     */
    public function numberOption($top, $start, $end, $step, $way='up')
    {
        $option = array();

        if ($top) {
            $option[0] = $top;
        }

        if ($way == 'up') {
            for ($i = $start; $i <= $end; $i += $step) {
                $option[$i] = $i;
            }
        } else {
            for ($i = $start; $end <= $i; $i -= $step) {
                $option[$i] = $i;
            }
        }

        return $option;
    }

    /**
     * セレクトボックの多言語対応オプション配列生成
     */
    public function selectOptionLang($options, $wordbook, $lang)
    {
        $selectOption = array();

        foreach ($options as $val => $word) {
            $selectOption[$val] = isset($wordbook[$word][$lang]) ? $wordbook[$word][$lang] : $word;
        }

        return $selectOption;
    }

    /**
     * チェックボックス
     *
     * @options 選択肢 array(key=>title)
     * @values 選択状態 array(key=>value)
     */
    public function checkBoxes($id, $class, $name, $options=array(), $values=array())
    {
        $id ?: $this->_randString();
        $class = empty($class) ? '' : sprintf(' class="%s"', $class);

        ob_start();

        $no = 1;
        foreach ($options as $key => $title) {
            $inName = empty($name) ? $key : sprintf('%s[%s]', $name, $key);
            $checked = isset($values[$key]) && $values[$key] == 1 ? ' checked="checked"' : '';
            echo sprintf('<label%s><input type="checkbox" id="%s-%d"%s name="%s" value="1"%s />%s</label>',
                $class, $id, $no++, $class, $inName, $checked, $title);
        }

        return ob_get_clean();
    }

    /**
     * チェックボックス
     */
    public function checkBox($id, $class, $name, $title, $value, $val='')
    {
        $for = '';
        if (!empty($id)) {
            $for = sprintf(' for="%s"', $id);
            $id = sprintf(' id="%s"', $id);
        }
        $class = empty($class) ? '' : sprintf(' class="%s"', $class);
        $name = empty($name) ? '' : sprintf(' name="%s"', $name);

        ob_start();

        echo sprintf('<input%s%s type="checkbox"%s value="%s"%s>', $id, $class, $name, $value, $this->setChecked($value, $val));
        echo sprintf('<label%s>%s</label>', $for, $title);

        return ob_get_clean();
    }

    /**
     * 一般的なラジオボタン
     */
    public function radioButton($id, $class, $name, $options, $val)
    {
        $class = empty($class) ? '' : sprintf(' class="%s"', $class);
        $name = empty($name) ? '' : sprintf(' name="%s"', $name);
        $attr = empty($attr) ? '' : sprintf(' %s="%s"', $attr, $attr);

        ob_start();

        foreach ($options as $key => $title) {
            $checked = $this->setChecked($key, $val);
            $idStr = sprintf("%s-%s", $id, $key);
            echo sprintf('<label for="%s"%s><input id="%s" type="radio"%s value="%s"%s>%s</label>',
                $idStr, $class, $idStr, $name, $key, $checked, $title);
        }

        return ob_get_clean();
    }

    // ランダム文字列を戻す
    private function _randString($number=8)
    {
        static $base = '0123456789abcdefghijklmnopqrstuvwxyz';

        $str = '';
        for ($i = 0; $i < $number; $i++) {
            $str .= substr($base, mt_rand(0, 35), 1);
        }

        return $str;
    }

    /**
     * 一般的なテキストボックス
     */
    public function textBox($id, $class, $name, $val, $suffix='')
    {
        $id = empty($id) ? '' : sprintf(' id="%s"', $id);
        $class = empty($class) ? '' : sprintf(' class="%s"', $class);
        $name = empty($name) ? '' : sprintf(' name="%s"', $name);

        ob_start();

        echo sprintf('<input type="text" %s%s%s value="%s">', $id, $class, $name, esc_html($val));
        echo $this->_suffix($suffix) . "\n";

        return ob_get_clean();
    }

    /**
     * 隠しテキストボックス
     */
    public function hiddenBox($id, $class, $name, $val)
    {
        $id = empty($id) ? '' : sprintf(' id="%s"', $id);
        $class = empty($class) ? '' : sprintf(' class="%s"', $class);
        $name = empty($name) ? '' : sprintf(' name="%s"', $name);

        ob_start();

        echo sprintf('<input type="hidden" %s%s%s value="%s">', $id, $class, $name, esc_html($val));
        echo "\n";

        return ob_get_clean();
    }

    /**
     * 一般的なテキストエリアボックス
     */
    public function textareaBox($id, $class, $name, $val, $suffix='', $attr='')
    {
        ob_start();

        echo sprintf('<textarea id="%s" class="%s" name="%s" %s>%s</textarea>%s', $id, $class, $name, $attr, esc_html($val), $suffix);
        echo $this->_suffix($suffix) . "\n";

        return ob_get_clean();
    }

    /**
     * オプション選択の設定
     *
     * @val1    needle
     * @val2    array
     */
    public function setSelected($val1, $val2)
    {
        return in_array($val1, $val2) ? ' selected="selected"' : '';
    }

    /**
     * ラジオボタン・チェックボックスの設定
     *
     */
    public function setChecked($val1, $val2)
    {
        return $val1 == $val2 ? ' checked="checked"' : '';
    }

    /**
     * ボタンの有効無効の設定
     */
    public function setDisabled($check)
    {
        return $check ? '' : ' disabled';
    }

    /**
     * 単位
     */
    private function _suffix($suffix)
    {
        if (!empty($suffix)) {
            return sprintf('<span class="form-addchr suffix">%s</span>', $suffix);
        }
        return '';
    }

    /**
     * 確認画面用埋込み項目一括変換
     *
     */
    public function embedCustomHtml($name, $customs)
    {
        $embedStr = '';

        foreach ($customs as $id => $custom) {
            switch ($custom['type']) {
                case 'time':
                    $embedStr .= $this->embedTime($name, $id, $custom['value']);
                    break;
                case 'check':
                case 'select':
                case 'radio':
                    $embedStr .= $this->embedCheck($name, $id, $custom['value'], $custom['options']);
                    break;
                case 'text':
                    $embedStr .= $this->embedText($name, $id, $custom['value']);
                    break;
            }
        }

        return $embedStr;
    }

    // テキストデータの埋込み
    public function embedText($name, $id, $val)
    {
        return sprintf('<input type="hidden" name="%s[%s]" value="%s">', $name, $id, esc_html($val)) . "\n";
    }

    // チェックボックス、ラジオボタン、セレクトボックスデータの埋込み
    public function embedCheck($name, $id, $val, $options)
    {
        $embedStr = '';
        $aval = explode(',', $val);

        foreach ($options as $idx => $item) {
            if (in_array($item, $aval)) {
                $embedStr .= sprintf('<input type="hidden" name="%s[%s][]" value="%d">', $name, $id, $idx) . "\n";
            }
        }

        return $embedStr;
    }

    // 時刻データの埋込み変換
    public function embedTime($name, $id, $val)
    {
        $time = explode(':', $val);

        $embedStr = sprintf('<input type="hidden" name="%s[%s][hour]" value="%d">', $name, $id, $time[0]) . "\n";
        $embedStr .= sprintf('<input type="hidden" name="%s[%s][minute]" value="%d">', $name, $id, $time[1]) . "\n";

        return $embedStr;
    }

    // 入力項目のエラー出力HTML
    public function errorField($msg)
    {
        if (!empty($msg)) {
            return sprintf('<div class="mts-error-field">%s</div>', $msg);
        }
        return '';
    }

    /**
     * 項目概要説明の出力
     */
    public function description($description)
    {
        return sprintf('<div class="mts-description">%s</div>', $description);
    }

    /**
     * 名称多言語入力項目
     */
    public function multiLangTitle($id, $class, $name, $vals, $lang)
    {
        ob_start();

        foreach ($lang as $langKey => $language) {
            $attrName = empty($name) ? "{$id}[{$langKey}]" : "{$name}[{$langKey}]";
            echo sprintf('<label class="input-language"><input id="%s-%s" class="%s" type="text" name="%s" value="%s">(%s)</label>',
                $id, $langKey, $class, $attrName, (isset($vals->$langKey) ? $vals->$langKey : ''), $language);
            echo "\n";
        }

        return ob_get_clean();
    }

    /**
     * デモ用のためのメール非送信の確認
     */
    public function demoMailCheck($lang='ja')
    {
        $demo = get_post_meta(get_the_ID(), 'mtsab_demo', true);

        ob_start();
        if ($demo == 1) :
?>
            <div id="mtsab-booking-demo" class="booking-box">
                <p><?php if ($lang == 'ja') : ?>
                        <span class="demo-message" style="color:red">デモ用サンプルのため確認メール送信は無効になっています。</span><br>
                        入力メールアドレスに確認メールを受け取る場合はチェックして下さい。<br>
                        <input type="hidden" name="mtsab_demo[mail]" value="0">
                        <label><input type="checkbox" name="mtsab_demo[mail]" value="1"> メールを受け取る</label>
                    <?php else : ?>
                        <span class="demo-message" style="color:red">As it is demonstration, the confirmation mail is not available.</span><br>
                        If you want to recieve the confirmation mail, check the following check box, please.<br>
                        <input type="hidden" name="mtsab_demo[mail]" value="0">
                        <label><input type="checkbox" name="mtsab_demo[mail]" value="1"> Send Confirmation Mail</label>
                    <?php endif; ?></p>
            </div>

<?php
        endif;

        return ob_get_clean();
    }

}