<?php
/**
 * MTS Car Booking フロントエンド予約処理のビュー
 *
 * @Filename    BookingCarView.php
 * @Author      S.Hayashi
 * @Code        2018-09-30 Ver.1.0.0
 */
namespace MTSCarBookingTrial\views;

use MTSCarBookingTrial\CarBooking;
use MTSCarBookingTrial\Config;
use MTSCarBookingTrial\BookingCar;
use MTSCarBookingTrial\lang\Translation;
use MTSCarBookingTrial\models\Calendar;
use MTSCarBookingTrial\models\Rental;
use MTSCarBookingTrial\models\Vehicle;
use MTSCarBookingTrial\models\Option;
use MTSCarBookingTrial\models\Schedule;
use MTSCarBookingTrial\models\Alternate;

class BookingCarView extends RentalCarView
{
    const PAGE_NAME = BookingCar::PAGE_NAME;

    const JS = 'js/booking-car.js';
    //const ASSISTANCE_JS = 'js/mts-assistance.js';
    const CSS = 'css/mtsrcb-booking-car.css';
    const LOADING_GIF = 'img/ajax-loader.gif';

    private $pageUrl;               // 予約登録・編集ページのURL

    private $oHelper = null;

    private $oBookingCar = null;

    /**
     * Constructor
     */
    public function __construct(BookingCar $oBookingCar)
    {
        $this->oBookingCar = $oBookingCar;

        $this->lang = $oBookingCar->lang;

        $this->oHelper = new FormHelper;

        // 予約フォームページのURL
        $this->pageUrl = get_permalink();

        // JavaScript 組み込み登録の設定
        $this->_enqueueScripts();
    }

    // JavaScript,Style headタグ内のロード登録
    private function _enqueueScripts()
    {
        // ローカライズデータ
        $local = array(
            'dateFormat' => Translation::$dateFormat[$this->lang],
            'yearMonthFormat' => Translation::$yearMonthFormat[$this->lang],
            'week' => Translation::$week[$this->lang],
            'monthF' => isset(Translation::$monthF[$this->lang]) ? Translation::$monthF[$this->lang] : array(),
            'monthM' => isset(Translation::$monthM[$this->lang]) ? Translation::$monthM[$this->lang] : array(),
            'lang' => $this->lang,
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce(self::PAGE_NAME),
        );

        $dir = dirname(__FILE__);

        wp_enqueue_script(self::PAGE_NAME, plugins_url(self::JS, $dir), array('jquery'));
        //wp_enqueue_script(self::PAGE_NAME . '-assistance', plugins_url(self::ASSISTANCE_JS, $dir), array('jquery'));
        wp_localize_script(self::PAGE_NAME, 'localform', $local);
        wp_enqueue_style(self::PAGE_NAME, plugins_url(self::CSS, $dir));

        $oSetting = Config::getSetting('customer');

        if (0 < $oSetting->zip_search) {
            CustomerView::enqueuePostcode($oSetting->zip_search);
        }
    }

    /**
     * 作業中止表示ページ
     */
    public function abortPage($messageId='', $url='')
    {
        ob_start();
?>
        <div id="mtsrcb-form-abort">
<?php
            $this->_outMessage();
            if ($messageId) {
                echo sprintf('<div id="mtsrcb-abort-text"><a href="%s">%s</a></div>',
                    $url, Translation::$bForm[$messageId][$this->lang]);
            }
?>
        </div">

<?php

        return ob_get_clean();
    }

    // メッセージを表示する
    private function _outMessage()
    {
        if ($this->oBookingCar->msgCode === '') {
            return;
        }

        $this->_dispMessage($this->oBookingCar->msgCode, $this->oBookingCar->msgSub, $this->oBookingCar->errflg);
    }


    /**
     * レンタル日時エントリーフォーム
     */
    public function entryForm(Rental $oRental, Vehicle $oVehicle, Calendar $oCalendar, $calendarView)
    {

        ob_start();
?>
        <div id="mtsrcb-entry-form" class="entry-page mtsrcb-page-block">
            <?php $this->_outMessage() ?>

            <div id="mtsrcb-entry-vehicle">
                <?php $this->_outEntryVehicle($oVehicle) ?>
            </div>

            <div id="mtsrcb-entry-date">
                <?php $this->_outDateEntryForm($oRental, $oVehicle) ?>
            </div>

            <div id="mtsrcb-entry-calendar">
                <?php $this->_dispCalendar($oCalendar, $calendarView) ?>
            </div>
        </div>

<?php

        return ob_get_clean();
    }

    // 予約対象のレンタカー
    private function _outEntryVehicle(Vehicle $oVehicle)
    {
        $vehicle = sprintf('<h3 class="rental-vehicle-name">%s</h3>', $oVehicle->name->{$this->lang});
        echo apply_filters('mtsrcb-booking-vehicle', $vehicle, $oVehicle, $this->lang);
    }

    // レンタル期間エントリーフォーム出力
    private function _outDateEntryForm(Rental $oRental, Vehicle $oVehicle)
    {
        // ショップ営業時間を出発・返却時間選択の範囲として条件設定する
        $rsvSetting = Config::getSetting('reserve');

        $param = array(
            'start' => intval($rsvSetting->opening / 3600),
            'end' => intval($rsvSetting->closing / 3600),
            'interval' => intval($rsvSetting->interval ? ($rsvSetting->interval % 60) : 1),
        );

?>
        <div class="action-row description"><?php echo Translation::$bForm['ENTRY_DATE_DESCRIPTION'][$this->lang] ?></div>
        <form id="mtsrcb-entry-form" method="post" action="<?php echo $this->pageUrl ?>">
            <div id="rental-start-box" class="rental-date-box">
                <label class="date-box-label" for="date-rent-start"><?php echo Translation::$bForm['ENTRY_START_DATE'][$this->lang] ?></label>
                <div class="date-box-date">
                    <input id="date-rent-start" type="text" class="date-entry focused" value="<?php echo $this->_formatDate($oRental->rent_start) ?>" readonly>
                </div>
                <div class="date-box-time">
                    <?php echo $this->oHelper->selectTime('','rental-time','rental[start]', $oRental->rent_start, $param) ?>
                </div>
            </div>
            <div id="rental-date-arrow">➤</div>
            <div id="rental-end-box" class="rental-date-box">
                <label class="date-box-label" for="date-rent-end"><?php echo Translation::$bForm['ENTRY_END_DATE'][$this->lang] ?></label>
                <div class="date-box-date">
                    <input id="date-rent-end" type="text" class="date-entry" value="<?php echo $this->_formatDate($oRental->rent_end) ?>" readonly>
                </div>
                <div class="date-box-time">
                    <?php echo $this->oHelper->selectTime('','rental-time','rental[end]', $oRental->rent_end, $param) ?>
                </div>
            </div>
            <div class="rental-action-row">
                <input id="rent-start" type="hidden" name="rental[start][date]" value="<?php echo $oRental->rent_start - $oRental->rent_start % 86400 ?>">
                <input id="rent-end" type="hidden" name="rental[end][date]" value="<?php echo $oRental->rent_end - $oRental->rent_end % 86400 ?>">
                <input id="vehicle-id" type="hidden" name="schedule[vehicle_id]" value="<?php echo $oVehicle->postId ?>">
                <input type="hidden" name="rental[rental_id]" value="<?php echo $oRental->rental_id ?>">
                <input type="submit" class="button-primary" name="new_rental" value="<?php echo Translation::$bForm['ENTRY_DATE_BUTTON'][$this->lang] ?>">
                <input id="rental-nonce" type="hidden" name="rental_nonce" value="<?php echo wp_create_nonce(self::PAGE_NAME) ?>">
            </div>
        </form>

<?php
    }

    // 日付選択カレンダー表示
    private function _dispCalendar(Calendar $oCalendar, $calendarView)
    {
        // タイトルとデータ
        $titleStr = $this->_dispDate(Translation::$yearMonthFormat[$this->lang], $oCalendar->monthTime);
        $titleData = sprintf('data-month_time="%d"', $oCalendar->monthTime);

        // リンクボタンと状態
        $prevVal = Translation::$leftArrow[$this->lang];
        $prevLink = $oCalendar->linkLast < $oCalendar->monthTime ? '' : ' disabled';
        $nextVal = Translation::$rightArrow[$this->lang];
        $nextLink = $oCalendar->monthTime < $oCalendar->linkFuture ? '' : ' disabled';

        // AJAXローディングアイコン画像のURL
        $iconUrl = plugins_url(self::LOADING_GIF, dirname(__FILE__));
?>
        <div class="booking-calendar">
            <div id="change-calendar-pain">
                <div id="calendar-link-prev" class="calendar-button" data-link_last="<?php echo $oCalendar->linkLast ?>">
                    <input id="change-prev" type="button" value="<?php echo $prevVal ?>" onclick="mtsrcb_booking_car.changeMonth(-1)"<?php echo $prevLink ?>>
                </div>
                <div id="calendar-link-title" class="calendar-title" <?php echo $titleData ?>><?php echo $titleStr ?></div>
                <div id="calendar-link-next" class="calendar-button" data-link_future="<?php echo $oCalendar->linkFuture ?>">
                    <input id="change-next" type="button" value="<?php echo $nextVal ?>" onclick="mtsrcb_booking_car.changeMonth(1)"<?php echo $nextLink ?>>
                </div>
            </div>
            <div id="schedule-calendar-pain" style="position:relative;">
                <div id="month-calendar">
                    <?php echo $calendarView ?>
                 </div>
                <div id="month-calendar-loading" style="display: none; position: absolute; top: 0px; left: 0px; width: 100%; height: 100%;">
                    <img src="<?php echo $iconUrl ?>" style="height:24px; width:24px; position:absolute; top:50%; left:50%; margin-top:-12px; margin-left:-12px">
                </div>
            </div>
        </div>
        <div id="schedule-calendar-end" class="clear"></div>

<?php
    }

    // 日付表示
    private function _dispDate($format, $time)
    {
        if ($this->lang === 'ja') {
            return date_i18n($format, $time);
        }

        return date($format, $time);
    }

    /**
     * レンタル品選択表示
     */
    public function rentalForm(Rental $oRental, Vehicle $oVehicle)
    {

        ob_start();
?>
        <div id="mtsrcb-rental-form" class="mtsrcb-page-block">
            <?php $this->_outMessage() ?>
            <form enctype="multipart/form-data" method="post" action="">

            <?php $this->_dispRentalPeriod($oRental, $oVehicle) ?>

            <?php $this->_dispOption($oRental, $oVehicle->options) ?>

            <?php $this->_customerView($oRental->oCustomer) ?>

            <?php $this->_dispUpload($oRental) ?>

            <div class="rental-form-block action-row">
                <input type="hidden" name="rental[rental_id]" value="<?php echo $oRental->rental_id ?>">
                <input type="submit" class="button-primary" name="check_form" value="<?php echo Translation::$fButton['CHECK'][$this->lang] ?>">
                <input type="submit" class="button-primary" name="return_entry" value="<?php echo Translation::$fButton['BACK'][$this->lang] ?>">
                <input id="rental-nonce" type="hidden" name="rental_nonce" value="<?php echo wp_create_nonce(self::PAGE_NAME) ?>">
                <input type="hidden" name="form_start" value="<?php echo current_time('timestamp') ?>">
            </div>

            </form>

            <?php $this->_dispGdprLink() ?>
        </div>

<?php

        return ob_get_clean();
    }

    // オプションを表示する
    private function _dispOption(Rental $oRental, $options)
    {
        if (empty($options)) {
            return;
        }
?>
        <div class="rental-form-block">
            <h3 class="rental-form-title"><?php echo Translation::$bForm['FORM_OPTION_TITLE'][$this->lang] ?></h3>
            <table id="mts-option-form" class="mts-form-table"><?php /*  class="rental-option-table" */ ?>
                <?php foreach ($oRental->optionSchedules as $oOptionSchedule) {
                    $this->_dispOptionForm($oOptionSchedule, $options[$oOptionSchedule->option_id]);
                } ?>
            </table>
        </div>

<?php
    }

    // オプション入力フォームを表示する
    private function _dispOptionForm(Schedule $oSchedule, Option $oOption)
    {
        $disabled = '';

        // 必須選択項目の場合
        if (0 < $oOption->type) {
            $disabled = ' disabled';
            $oSchedule->select = 1;
        }

?>
        <tr>
            <th><?php echo sprintf('<label for="rental-option-yes-%d">%s</label>',
                $oSchedule->option_id, $oOption->name->{$this->lang}) ?></th>
            <td class="option-select"><?php
                if ($oSchedule->unavailable) {
                    echo Translation::$bForm['FORM_OPTION_OUT'][$this->lang];
                } else {
                    echo sprintf('<label class="rental-option-label"><input id="rental-option-no-%d" type="radio" name="option[schedule][%d]" value="0"%s%s> %s</label>',
                      $oSchedule->option_id, $oSchedule->option_id, $this->oHelper->setChecked(0, $oSchedule->select), $disabled, Translation::$bForm['FORM_OPTION_UNNEED'][$this->lang]);
                    echo sprintf('<label class="rental-option-label"><input id="rental-option-yes-%d" type="radio" name="option[schedule][%d]" value="1"%s> %s</label>',
                        $oSchedule->option_id, $oSchedule->option_id, $this->oHelper->setChecked(1, $oSchedule->select), Translation::$bForm['FORM_OPTION_RENTAL'][$this->lang]);
                } ?></td>
        </tr>

<?php
    }

    /**
     * レンタル予約申込確認表示
     */
    public function rentalOrder(Rental $oRental, Vehicle $oVehicle)
    {
        $settingForm = Config::getSetting('form');

        ob_start();
?>
        <div id="mtsrcb-order-list" class="mtsrcb-page-block list-page">
            <?php $this->_outMessage() ?>

            <?php $this->_dispRentalPeriod($oRental, $oVehicle) ?>

            <?php $this->_dispOptionList($oRental->optionSchedules, $oVehicle->options) ?>

            <?php $this->_dispCharge($oRental, $oVehicle) ?>

            <?php $this->_customerView($oRental->oCustomer, 'confirmation') ?>

            <?php $this->_dispUploadState($oRental) ?>

            <?php $this->_dispGdprLink() ?>

            <form method="post" action="">
                <div class="rental-form-block">
                    <?php if (0 < $settingForm->concent) {
                        echo '<div id="rental-agreement-boock">';
                        echo '<input type="hidden" name="rental[acceptance]" value="0">';
                        echo '<input id="confirm-concent" type="checkbox" name="rental[acceptance]" value="1">';
                        echo sprintf('<label class="confirm-label" for="confirm-concent">%s</label>', Translation::$bForm['CONFIRMATION_AGREEMENT'][$this->lang]);
                        echo "</div>\n";
                    } ?>
                </div>

                <div class="rental-form-block action-row">
                    <input type="hidden" name="rental[rental_id]" value="<?php echo $oRental->rental_id ?>">
                    <input type="submit" class="button-primary" name="booking_rental" value="<?php echo Translation::$fButton['BOOKING'][$this->lang] ?>">
                    <input type="submit" class="button-primary" name="return_form" value="<?php echo Translation::$fButton['BACK'][$this->lang] ?>">
                    <input type="submit" class="button-primary" name="abort_rental" value="<?php echo Translation::$fButton['ABORT'][$this->lang] ?>">
                    <input id="rental-nonce" type="hidden" name="rental_nonce" value="<?php echo wp_create_nonce(self::PAGE_NAME) ?>">
                    <input type="hidden" name="form_start" value="<?php echo current_time('timestamp') ?>">
                </div>
            </form>

        </div>

<?php

        return ob_get_clean();
    }

    /**
     * 予約終了表示
     */
    public function finishPage(Rental $oRental, Vehicle $oVehicle, Alternate $oAlternate)
    {
        // 予約終了メッセージ
        $finMessage = $oAlternate->replaceVar(Translation::$bookingThanks[$this->lang]);

        // 決済情報を取得する
        $payment = $oRental->paymentAvailable();

        $actionUrl = $payButton = '';
        if ($payment) {
            $payButton = sprintf('<input type="submit" class="button-primary" name="reserve_payment" value="%s">',
                Translation::$fButton['PAY'][$this->lang]);
/*
            // 決済ページのURL
            $paymentPage = get_page_by_path(CarBooking::PAYMENT_PAGE);
            if ($paymentPage) {
                $actionUrl = get_permalink($paymentPage);
            }
*/
        }

        ob_start();
?>
        <div id="mtsrcb-finish-list" class="mtsrcb-page-block list-page">
            <?php $this->_outMessage() ?>

            <p><?php echo $finMessage ?></p>

            <?php $this->_dispRentalPeriod($oRental, $oVehicle) ?>

            <?php $this->_dispOptionList($oRental->optionSchedules, $oVehicle->options) ?>

            <?php $this->_dispCharge($oRental, $oVehicle) ?>

            <?php $this->_customerView($oRental->oCustomer, 'confirmation') ?>

            <?php $this->_dispUploadState($oRental) ?>

            <?php $this->_dispGdprLink() ?>

            <form method="post" action="<?php echo $actionUrl ?>">
                <div class="rental-form-block action-row">
                    <input type="hidden" name="rental[rental_id]" value="<?php echo $oRental->rental_id ?>">
                    <?php echo $payButton ?>
                    <input type="button" value="<?php echo Translation::$fButton['PRINT'][$this->lang] ?>" onClick="window.print();">
                    <input id="rental-nonce" type="hidden" name="rental_nonce" value="<?php echo wp_create_nonce(self::PAGE_NAME) ?>">
                    <input type="hidden" name="form_start" value="<?php echo current_time('timestamp') ?>">
                </div>
            </form>

        </div>

<?php
        return ob_get_clean();
    }

}