<?php
/**
 * MTS Car Booking レンタルスケジュールデータ
 *
 * @Filename    Schedule.php
 * @Author      S.Hayashi
 * @Code        2018-07-25 Ver.1.0.0
 */
namespace MTSCarBookingTrial\models;

use MTSCarBookingTrial\Config;
use MTSCarBookingTrial\CustomPost;

class Schedule
{
    const MODULE = 'system';
    const MODEL = 'schedule';
    const VERSION = '1.0';

    // テーブル名
    private $tableName = '';

    // レンタルデータ
    private $data = array(
        'schedule_id' => 0,
        'rental_id' => 0,           // レンタル予約ID
        'vehicle_id' => 0,          // 車両のpost ID
        'option_id' => 0,           // オプションのpost ID
        'user_id' => 0,             // ユーザーID
        'rent_start' => 0,          // レンタル開始日時
        'rent_end' => 0,            // レンタル終了日時
        'odo_start' => 0,           // レンタル前走行距離
        'odo_end' => 0,             // レンタル後走行距離
        'charge' => 0,              // 料金
        'note' => '',               // 摘要
        'updated' => '',
        'created' => '',
    );


    // レンタル予約時のオプション選択チェック
    public $select = false;         // 選択済み

    // レンタル予約時のレンタル許可チェック
    public $unavailable = false;    // 貸出中

    // レンタル予約時の料金計算結果;
    //public $charge = '';            // 料金

    public $errCode = '';
    public $errSub = '';

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->tableName = Config::getTableName(self::MODEL);
    }

    /**
     * 指定されたIDのスケジュールデータを戻す
     */
    public static function readSchedule($scheduleId)
    {
        global $wpdb;

        $tableName = Config::getTableName(self::MODEL);

        $data = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$tableName} WHERE schedule_id=%d", $scheduleId), ARRAY_A);

        if ($data) {
            $oSchedule = new Schedule;
            $oSchedule->setData($data);
            return $oSchedule;
        }

        return false;
    }

    /**
     * 指定されたレンタルIDのデータオブジェクトを配列で戻す
     */
    public static function findRental($rentalId)
    {
        global $wpdb;

        $tableName = Config::getTableName(self::MODEL);

        $sql = $wpdb->prepare(
            "SELECT * FROM {$tableName} WHERE rental_id=%d ORDER BY schedule_id ASC", $rentalId);

        $data = $wpdb->get_results($sql, ARRAY_A);

        $items = array();

        if ($data) {
            foreach ($data as $scheduleData) {
                $oSchedule = new Schedule;
                $oSchedule->setData($scheduleData);

                // 車両の場合は先頭に配列の先頭に置く
                if (0 < $oSchedule->vehicle_id) {
                    $items = array($oSchedule->schedule_id => $oSchedule) + $items;
                } else {
                    $items = $items + array($oSchedule->schedule_id => $oSchedule);
                }
            }
        }

        return $items;
    }

    /**
     * スケジュールを新規追加する
     */
    public function addSchedule()
    {
        global $wpdb;

        // 新規登録日時
        $this->created = $this->updated = current_time('mysql');

        $result = $wpdb->insert(
            $this->tableName,
            array(
                'rental_id' => $this->rental_id,
                'vehicle_id' => $this->vehicle_id,
                'option_id' => $this->option_id,
                'user_id' => $this->user_id,
                'rent_start' => $this->rent_start,
                'rent_end' => $this->rent_end,
                'odo_start' => $this->odo_start,
                'odo_end' => $this->odo_end,
                'charge' => $this->charge,
                'note' => $this->note,
                'updated' => $this->updated,
                'created' => $this->created
            ),
            array(
                '%d', '%d', '%d', '%d', '%d', '%d', '%d', '%d', '%s', '%s', '%s', '%s'
            )
        );

        if ($result) {
            $this->schedule_id = $wpdb->insert_id;
            $this->_clearError();
            return true;
        }

        return $this->_setError('FAILED_INSERT_SCHEDULE', sprintf('%d:%d', $this->vehicle_id, $this->option_id));
    }

    /**
     * スケジュールデータを保存する
     */
    public function saveSchedule()
    {
        global $wpdb;

        // 更新保存日時
        $this->updated = current_time('mysql');

        $res = $wpdb->update(
            $this->tableName,
            array(
                'rental_id' => $this->rental_id,
                'vehicle_id' => $this->vehicle_id,
                'option_id' => $this->option_id,
                'user_id' => $this->user_id,
                'rent_start' => $this->rent_start,
                'rent_end' => $this->rent_end,
                'odo_start' => $this->odo_start,
                'odo_end' => $this->odo_end,
                'charge' => $this->charge,
                'note' => $this->note,
                'updated' => $this->updated,
            ),
            array('schedule_id' => $this->schedule_id),
            array(
                '%d', '%d', '%d', '%d', '%d', '%d', '%d', '%d', '%s', '%s', '%s'
            ),
            array('%d')
        );

        if (!$res) {
            return $this->_setError('FAILED_UPDATE_SCHEDULE', $this->schedule_id);
        }

        $this->_clearError();
        return $res;
    }

    /**
     * 料金データを更新する
     */
    public function updateCharge()
    {
        global $wpdb;

        // 更新保存日時
        $this->updated = current_time('mysql');

        $res = $wpdb->update(
            $this->tableName,
            array(
                'charge' => $this->charge,
                'updated' => $this->updated,
            ),
            array('schedule_id' => $this->schedule_id),
            array(
                '%s', '%s'
            ),
            array('%d')
        );

        if (!$res) {
            return $this->_setError('FAILED_UPDATE_SCHEDULE', $this->schedule_id);
        }

        $this->_clearError();
        return $res;
    }

    /**
     * スケジュールデータを削除する
     */
    public function deleteSchedule()
    {
        global $wpdb;

        $res = $wpdb->delete(
            $this->tableName,
            array('schedule_id' => $this->schedule_id),
            array('%d')
        );

        if (!$res){
            return $this->_setError('FAILED_DELETE_SCHEDULE', $this->schedule_id);
        }

        return $res;
    }

    /**
     * 走行メーターのデータを更新する
     */
    public function updateOdo($start, $end)
    {
        global $wpdb;

        $updated = current_time('mysql');

        $res = $wpdb->update(
            $this->tableName,
            array(
                'odo_start' => $start,
                'odo_end' => $end,
                'updated' => $updated,
            ),
            array('schedule_id' => $this->schedule_id),
            array('%d', '%d', '%s'),
            array('%d')
        );

        if ($res) {
            $this->odo_start = $start;
            $this->odo_end = $end;
            $this->updated = $updated;
        } else {
            $this->setError('FAILED_UPDATE_SCHEDULE_ODO', $this->schedule_id);
        }

        return $res;
    }

    // エラーをセットする
    private function _setError($errCode, $errSub='')
    {
        $this->errCode = $errCode;
        $this->errSub = $errSub;

        return false;
    }

    // エラーをクリアする
    private function _clearError()
    {
        $this->errCode = $errSub = '';
    }

    /**
     * データを戻す
     */
    public function getData()
    {
        return $this->data;
    }

    /**
     * データをオブジェクトにセットする
     */
    public function setData($data)
    {
        $this->data = $data;
    }

    /**
     * プロパティから読み出す
     */
    public function __get($key)
    {
        if (array_key_exists($key, $this->data)) {
            if (is_array($this->data[$key]) && $key !== 'table') {
                return (object) $this->data[$key];
            }
            return $this->data[$key];
        }

        if (isset($this->$key)) {
            return $this->$key;
        }

        $trace = debug_backtrace();
        trigger_error(sprintf(
            "Undefined property: '%s&' in %s on line %d, E_USER_NOTICE",
            $key, $trace[0]['file'], $trace[0]['line']
        ));

        return null;
    }

    /**
     * プロパティをセットする
     */
    public function __set($key, $value)
    {
        if (array_key_exists($key, $this->data)) {
            if (is_array($value)) {
                foreach ($value as $item => $val) {
                    $this->data[$key][$item] = $val;
                }
            } else {
                $this->data[$key] = $value;
            }
        } else {
            $this->$key = $value;
        }

        return $value;
    }

    /**
     * レンタルスケジュールテーブルの生成とアップデート
     */
    public static function installTable(&$version, $list)
    {
        // スケジュールテーブル名を取得する
        $tableName = Config::getTableName(self::MODEL);

        // テーブルが登録されていない、またはバージョンが異なれば更新する
        if (array_key_exists(self::MODEL, $version) && in_array($tableName, $list)) {
            if ($version[self::MODEL] === self::VERSION) {
                return false;
            }
        }

        $sql = "CREATE TABLE $tableName (
            schedule_id int(11) unsigned NOT NULL AUTO_INCREMENT,
            rental_id int(11) DEFAULT '0',
            vehicle_id bigint(11) DEFAULT '0',
            option_id bigint(11) DEFAULT '0',
            user_id int(11) DEFAULT '0',
            rent_start bigint(12) DEFAULT '0',
            rent_end bigint(12) DEFAULT '0',
            odo_start int(11) DEFAULT '0',
            odo_end int(11) DEFAULT '0',
            charge varchar(12) DEFAULT '',
            note text,
            updated datetime DEFAULT NULL,
            created datetime DEFAULT NULL,
            PRIMARY KEY  (schedule_id),
            KEY rental_id (rental_id),
            KEY vehicle_id (vehicle_id),
            KEY option_id (option_id),
            KEY rent_start (rent_start),
            KEY rent_end (rent_end)
        ) DEFAULT CHARSET=utf8mb4;";

        dbDelta($sql);

        // バージョンデータの更新
        $version[self::MODEL] = self::VERSION;

        return true;
    }

    /**
     * 車両、オプション名参照リストを取得する
     */
    public static function rentalNameList()
    {
        global $wpdb;

        $orderby = 'ASC';

        $sql = $wpdb->prepare(
            "SELECT ID,post_title,post_type "
            . "FROM $wpdb->posts "
            . "WHERE post_type in (%s, %s) AND post_status=%s "
            . "ORDER BY menu_order $orderby", CustomPost::VEHICLE, CustomPost::OPTION, 'publish');

        $data = $wpdb->get_results($sql, ARRAY_A);

        $list = array();
        if ($data) {
            foreach ($data as $item) {
                $list[$item['ID']] = array(
                    'name' => $item['post_title'],
                    'type' => str_replace(Config::DOMAIN . '_', '', $item['post_type']),
                );
            }
        }

        return $list;
    }

    /**
     * 指定期間中の車両貸し出しリストを検索する
     */
    public static function findRentalVehicle($starttime, $endtime, $vehicleIds=array())
    {
        global $wpdb;

        $tableName = Config::getTableName(self::MODEL);

        // 車両指定
        $condition = 'vehicle_id>0';
        if ($vehicleIds) {
            $condition = sprintf('vehicle_id IN (%s)', implode(',', $vehicleIds));
        }

        $sql = $wpdb->prepare(
            "SELECT schedule_id,rental_id,vehicle_id,rent_start,rent_end,note "
            . "FROM {$tableName} "
            . "WHERE {$condition} AND "
            . "((rent_start>=%d AND rent_start<%d) OR (rent_end>=%d AND rent_end<%d)) "
            . "ORDER BY vehicle_id ASC,rent_start ASC",
            $starttime, $endtime, $starttime, $endtime);

        $data = $wpdb->get_results($sql, ARRAY_A);

        $rental = array();
        if ($data) {
            foreach ($data as $schedule) {
                $rental[$schedule['vehicle_id']][] = $schedule;
            }
        }

        return $rental;
    }

    /**
     * 指定期間中のオプション貸し出しリストを検索する
     */
    public static function findRentalOption($starttime, $endtime, $optionIds=array())
    {
        global $wpdb;

        $tableName = Config::getTableName(self::MODEL);

        // オプション指定
        $condition = 'option_id>0';
        if ($optionIds) {
            $condition = sprintf('option_id IN (%s)', implode($optionIds));
        }

        $sql = $wpdb->prepare(
            "SELECT schedule_id,rental_id,option_id,rent_start,rent_end,note "
            . "FROM {$tableName} "
            . "WHERE {$condition} AND "
            . "((rent_start>=%d AND rent_start<%d) OR (rent_end>=%d AND rent_end<%d)) "
            . "ORDER BY vehicle_id ASC,rent_start ASC",
            $starttime, $endtime, $starttime, $endtime);

        $data = $wpdb->get_results($sql, ARRAY_A);

        $rental = array();
        if ($data) {
            foreach ($data as $schedule) {
                $rental[$schedule['option_id']][] = $schedule;
            }
        }

        return $rental;
    }

    /**
     * オプション在庫数量確認
     */
    public function rentalAvailable(Option $oOption=null)
    {
        // オプションデータを取得する
        if (empty($oOption)) {
            $oOption = Option::readOption($this->option_id);
            if (empty($oOption)) {
                return false;
            }
        }

        // 必須選択、貸出管理なしの場合は有効とする
        if ($oOption->type || $oOption->inventory<=0) {
            return true;
        }

        // レンタル開始日0時と終了日24時のUnix Time
        $startDaytime = strtotime(date('Y-n-j', $this->rent_start));
        $endDaytime = strtotime(date('Y-n-j', $this->rent_end + 86400));

        // 期間中のレンタルスケジュールを取得する
        $rentals = self::findRentalOption($startDaytime, $endDaytime, array($oOption->postId));
        if ($rentals) {
            $rentals = $rentals[$oOption->postId];
        }

        // 貸出がなければ許可
        if (empty($rentals)) {
            return true;
        }

        // 貸出スケジュールを初期化する
        $schedules = array();
        for ($i = 0; $i < $oOption->quantity; $i++) {
            $schedules[$i][] = array(0, 0);
        }

        // レンタルスケジューリングする
        foreach ($rentals as $rental) {
            // 登録データが当該オブジェクトならスケジューリングから外す
            if ($rental['schedule_id'] == $this->schedule_id) {
                continue;
            }

            $rentStart = strtotime(date('Y-n-j', $rental['rent_start']));
            $rentEnd = strtotime(date('Y-n-j'), $rental['rent_end']);

            for ($i = 0; $i < $oOption->quantity; $i++) {
                $last = end($schedules[$i]);
                if ($last[1] < $rentStart) {
                    $schedules[$i][] = array($rentStart, $rentEnd);
                    break;
                }
            }

        }

        // 在庫があれば予約可能とする
        for ($i = 0; $i < $oOption->quantity; $i++) {
            if (count($schedules[$i]) <= 1) {
                $this->unavailable = false;
                return true;
            }
        }

        // 在庫なし
        $this->unavailable = true;
        return false;
    }

    /**
     * 車両の予約状況を確認する
     */
    public function vehicleAvailable()
    {
        // レンタル開始日0時と終了日24時のUnix Time
        $startDaytime = strtotime(date('Y-n-j', $this->rent_start));
        $endDaytime = strtotime(date('Y-n-j', $this->rent_end + 86400));

        // 車両の予約を検索する
        $rentals = self::findRentalVehicle($startDaytime, $endDaytime, array($this->vehicle_id));

        $this->unavailable = false;

        if (empty($rentals[$this->vehicle_id])) {
            return true;
        }

        // スケジュールIDが異なる場合は期間中の貸出有り
        foreach($rentals[$this->vehicle_id] as $rental) {
            if ($rental['schedule_id'] != $this->schedule_id) {
                $this->unavailable = true;
                return false;
            }
        }

        return true;
    }

}
