<?php
/**
 * MTS Car Booking オプションマスタ(カスタム投稿タイプ)
 *
 * @Filename    Option.php
 * @Author      S.Hayashi
 * @Code        2018-07-11 Ver.1.0.0
 */
namespace MTSCarBookingTrial\models;

use MTSCarBookingTrial\Config;
use MTSCarBookingTrial\CustomPost;

class Option
{
    const NUMBER_OF_IMAGES = 2;

    // 名称
    private static $optionName = array(
        'ja' => '',                 // 日本語
        'en' => '',                 // 英語
    );

    // ポストメタデータカラム
    private static $postmetas = array('name', 'type', 'inventory', 'quantity', 'calculation', 'charge_id', 'image');

    // オプションデータ
    private $data = array(
        'postId' => 0,              // オプションのポストID
        'name' => array(),          // 表示名称
        'type' => 0,                // 選択種別(0:選択 1:必須)
        'inventory' => 0,           // 貸出管理(0:しない 1:する)
        'quantity' => 1,            // 数量
        'calculation' => 0,         // 料金計算(0:しない 1:する)
        'charge_id' => 0,           // 料金表ポストID
        'summary' => '',            // 概要
        'image' => array(),         // オプション画像
        'oPost' => null,            // カスタムポストデータ
    );

    // 料金データ
    public $oCharge = null;

    public $errCode = '';
    public $errSub = '';

    // ポストメタデータのキーを戻す
    protected static function _metaKey($column)
    {
        return sprintf('%s_%s', CustomPost::OPTION, $column);
    }

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->data['name'] = self::$optionName;
        for ($i = 0; $i < self::NUMBER_OF_IMAGES; $i++) {
            $this->data['image'][$i] = '';
        }
    }

    /**
     * オプションデータを取得する
     */
    public static function readOption($postId)
    {
        $oOption = new Option;

        if (is_numeric($postId)) {
            $oOption->oPost = get_post($postId);
        } else {
            $oOption->oPost = $postId;
        }

        if ($oOption->oPost) {
            $oOption->postId = $oOption->oPost->ID;
            $oOption->_readPostmeta();
        }

        // 料金データの読み込み
        if (0 < $oOption->charge_id) {
            $oOption->oCharge = Charge::readCharge($oOption->charge_id);
        }

        return $oOption;
    }

    // ポストメタデータを読み込む
    private function _readPostmeta()
    {
        foreach ($this->data as $column => $data) {
            if (in_array($column, self::$postmetas)) {
                $metaVal = get_post_meta($this->postId, self::_metaKey($column), true);

                if (!empty($metaVal)) {
                    $this->data[$column] = $metaVal;
                }
            }
        }
    }

    /**
     * ポストメタデータを保存する
     */
    public function savePostmeta()
    {
        foreach ($this->data as $column => $data) {
            if (!in_array($column, self::$postmetas)) {
                continue;
            }

            $metaKey = self::_metaKey($column);

            if (!add_post_meta($this->postId, $metaKey, $data, true)) {
                $oldData = get_post_meta($this->postId, $metaKey, true);
                if ($oldData != $data) {
                    if (!update_post_meta($this->postId, $metaKey, $data)) {
                        return $this->_setError('FAILED_UPDATE_POSTMETA', $metaKey);
                    }
                }
            }
        }

        return true;
    }

    /**
     * ポストメタデータを削除する
     */
    public function deletePostmeta()
    {
        foreach ($this->data as $column => $data) {
            if (in_array($column, self::$postmetas)) {
                $metaKey = self::_metaKey($column);

                if (!delete_post_meta($this->postId, $metaKey)) {
                    return $this->_setError('FAILED_DELETE_POSTMETA', $metaKey);
                }
            }
        }

        return true;
    }

    // エラー終了
    private function _setError($errCode, $sub)
    {
        $this->errCode = $errCode;
        $this->errSub = $sub;

        return false;
    }

    /**
     * データを戻す
     */
    public function getData()
    {
        return $this->data;
    }

    /**
     * データを一括セットする
     */
    public function setData($data)
    {
        return $this->data = $data;
    }

    /**
     * プロパティから読み出す
     */
    public function __get($key)
    {
        if (array_key_exists($key, $this->data)) {
            if (is_array($this->data[$key]) && $key !== 'image') {
                return (object) $this->data[$key];
            }
            return $this->data[$key];
        }

        if (isset($this->$key)) {
            return $this->$key;
        }

        $trace = debug_backtrace();
        trigger_error(sprintf(
            "Undefined property: '%s&' in %s on line %d, E_USER_NOTICE",
            $key, $trace[0]['file'], $trace[0]['line']
        ));

        return null;
    }

    /**
     * プロパティをセットする
     */
    public function __set($key, $value)
    {
        if (array_key_exists($key, $this->data)) {
            if (is_array($value)) {
                foreach ($value as $item => $val) {
                    $this->data[$key][$item] = $val;
                }
            } else {
                $this->data[$key] = $value;
            }
        } else {
            $this->$key = $value;
        }

        return $value;
    }

    /**
     * オプション一覧を取得する
     *
     * return array(array(term_id => name), ...)
     */
/*
    public static function findOptionList($params=array('inventory'=>-1))
    {
        global $wpdb;

        $order = 'ASC';

        $inventory = '';

        // 在庫管理を有無
        if (0 <= $params['inventory']) {
            $inventory = sprintf(" AND m1.meta_value='%d'", 'inventory', $params['inventory']);
        }

        $sql = $wpdb->prepare(
            "SELECT ID,post_title,m1.meta_value AS inventory "
            . "FROM {$wpdb->posts} "
            . "LEFT JOIN {$wpdb->postmeta} AS m1 ON ID=m1.post_id AND m1.meta_key=%s "
            . "WHERE post_type=%s AND post_status=%s{$inventory} "
            . "ORDER BY menu_order {$order}", self::_metaKey('inventory'), CustomPost::VEHICLE, 'publish');

        $data = $wpdb->get_results($sql, ARRAY_A);

        $options = array();
        if ($data) {
            foreach($data as $option) {
                $options[$option['ID']] = array('name' => $option['post_title'], 'inventory' => $option['inventory']);
            }
        }

        return $options;
    }
*/

    /**
     * オプション名一覧を取得する
     */
    public static function listOption()
    {
        global $wpdb;

        $order = 'ASC';

        $sql = $wpdb->prepare(
            "SELECT ID,post_title,m1.meta_value AS inventory "
            . "FROM {$wpdb->posts} "
            . "LEFT JOIN {$wpdb->postmeta} AS m1 ON ID=m1.post_id AND m1.meta_key=%s "
            . "WHERE post_type=%s AND post_status=%s "
            . "ORDER BY m1.meta_value ASC, menu_order {$order}", self::_metaKey('inventory'), CustomPost::OPTION, 'publish');

        $data = $wpdb->get_results($sql, ARRAY_A);

        return self::_dataList($data);
    }

    // 読み込みデータを一覧リストにして戻す
    private static function _dataList($data)
    {
        $options = array();

        if ($data) {
            foreach ($data as $option) {
                $options[$option['ID']] = array(
                    'post_title' => $option['post_title'],
                    'inventory' => $option['inventory'],
                );
            }
        }

        return $options;
    }

    /**
     * 特定車種のオプション名一覧を取得する
     */
    public static function listOptionByType($term_id)
    {
        global $wpdb;

        $order = 'ASC';

        $sql = $wpdb->prepare(
            "SELECT ID,post_title,m1.meta_value AS inventory "
            . "FROM {$wpdb->posts} LEFT JOIN {$wpdb->postmeta} AS m1 ON ID=m1.post_id AND m1.meta_key=%s "
            . "WHERE ID IN "
            . "(SELECT object_id FROM {$wpdb->term_relationships} WHERE term_taxonomy_id IN "
            . "(SELECT term_taxonomy_id FROM {$wpdb->term_taxonomy} WHERE term_id=%d)) "
            . "AND post_type=%s AND post_status=%s "
            . "ORDER BY m1.meta_value ASC, menu_order {$order}", self::_metaKey('inventory'), $term_id, CustomPost::OPTION, 'publish');

        $data = $wpdb->get_results($sql, ARRAY_A);

        return self::_dataList($data);
    }


}
