<?php
/**
 * MTS Car Booking 画像データ
 *
 * @Filename    Graphics.php
 * @Author      S.Hayashi
 * @Code        2018-12-04 Ver.1.0.0
 */
namespace MTSCarBookingTrial\models;

use MTSCarBookingTrial\Config;

class Graphics
{
    const MODULE = 'system';
    const MODEL = 'graphics';
    const VERSION = '1.0';

    // テーブル名
    private $tableName = '';

    // 画像データ
    private $data = array(
        'graphics_id' => 0,
        'rental_id' => 0,           // レンタル予約ID
        'type' => '',               // データ種別
        'filename' => '',           // ファイル名
        'extension' => '',          // 識別子
        'image' => '',              // 画像データ
        'updated' => '',
        'created' => '',
    );

    public $errCode = '';
    public $errSub = '';

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->tableName = Config::getTableName(self::MODEL);
    }

    /**
     * アップロードファイルを読込む
     */
    public function retrieveFile($type, $files)
    {
        // エラーチェック
        if ($files['error'] !== UPLOAD_ERR_OK) {
            return $this->_setError($this->phpError($files['error']));
        }

        // データ種別
        $this->type = $type;

        // ファイル名
        $finfo = pathinfo($files['name']);
        $this->filename = $finfo['basename'];

        // 拡張子
        $ext = strtolower($finfo['extension']);
        switch ($ext) {
            case 'jpg':
            case 'jpeg':
                $this->extension = 'jpeg';
                break;
            case 'png':
                $this->extension = $ext;
                break;
            case 'gif':
                $this->extension = $ext;
                break;
            default:
                $this->extension = '';
                break;
        }

        // タイプ確認
        $iType = explode('/', $files['type']);
        if ($iType[1] !== $this->extension) {
            return $this->_setError('INVALID_IMAGE_TYPE', $this->filename);
        }

        $this->image = file_get_contents($files['tmp_name']);
        if ($this->image === false) {
            return $this->_setError('DISAPPEAR_TEMPORARY_IMAGE');
        }

        return true;
    }

    /**
     * 指定されたIDの画像データを戻す
     */
    public static function readGraphics($graphicsId)
    {
        global $wpdb;

        $tableName = Config::getTableName(self::MODEL);

        $data = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$tableName} WHERE grapchis_id=%d", $graphicsId), ARRAY_A);

        if ($data) {
            $oGraphics = new Graphics;
            $oGraphics->setData($data);
            return $oGraphics;
        }

        return false;
    }

    /**
     * 指定されたタイプの既存種別データを読込む
     */
    public function findGraphics($rentalId, $type)
    {
        global $wpdb;

        $sql = $wpdb->prepare(
            "SELECT * FROM {$this->tableName} WHERE rental_id=%d AND type=%s", $rentalId, $type);

        $data = $wpdb->get_row($sql, ARRAY_A);

        if ($data) {
            $this->setData($data);
            return true;
        }

        return false;
    }

    /**
     * 画像データを新規追加する
     */
    public function addGraphics()
    {
        global $wpdb;

        // 新規登録日時
        $this->created = $this->updated = current_time('mysql');

        $result = $wpdb->insert(
            $this->tableName,
            array(
                'rental_id' => $this->rental_id,
                'type' => $this->type,
                'filename' => $this->filename,
                'extension' => $this->extension,
                'image' => $this->image,
                'updated' => $this->updated,
                'created' => $this->created
            ),
            array(
                '%d', '%s', '%s', '%s', '%s', '%s', '%s'
            )
        );

        if ($result) {
            $this->graphics_id = $wpdb->insert_id;
            //$ret = $this->_updateImage();
            //var_dump($ret);
            $this->_clearError();
            return true;
        }

        return $this->_setError('FAILED_INSERT_GRAPHICS', sprintf('%d:%d', $this->rental_id, $this->type));
    }
/*
    // BLOBデータを保存する
    private function _updateImage()
    {
        global $wpdb;
        $pdo = null;

        //var_dump($wpdb);
        //echo sprintf('mysql:host=%s,dbname=%s,charset=utf8', $wpdb->dbhost, $wpdb->dbname);
        //echo sprintf(':%s', $wpdb->dbpassword);

        $conStr = sprintf('mysql:host=%s;dbname=%s;charset=utf8', $wpdb->dbhost, $wpdb->dbname);
        var_dump($conStr);
        var_dump($wpdb->dbuser);
        var_dump($wpdb->dbpassword);


        try {
            $pdo = new \PDO($conStr, $wpdb->dbuser, $wpdb->dbpassword);
        } catch (\PDOException $e) {
            die($e->getMessage());
            return $this->_setError($e->getMessage());
        }

        $sql = "UPDATE {$this->tableName} SET image=:data,updated=:updated WHERE graphics_id=:id";

        var_dump($sql);
        var_dump($this->image);

        $stmt = $pdo->prepare($sql);

        $stmt->bindValue(':data', $this->image, \PDO::PARAM_STR);
        $stmt->bindValue(':id', $this->graphics_id);
        $stmt->bindValue(':updated', current_time('mysql'));

        try {
            $stmt->execute();
        } catch (\PDOException $e) {
            die($e->getMessage());
        }

    }
*/


    /**
     * 画像データを保存する
     */
    public function updateGraphics()
    {
        global $wpdb;

        // 更新保存日時
        $this->updated = current_time('mysql');

        $res = $wpdb->update(
            $this->tableName,
            array(
                'rental_id' => $this->rental_id,
                'type' => $this->type,
                'filename' => $this->filename,
                'extension' => $this->extension,
                'image' => $this->image,
                'updated' => $this->updated,
            ),
            array('graphics_id' => $this->graphics_id),
            array(
                '%d', '%s', '%s', '%s', '%s', '%s'
            ),
            array('%d')
        );

        if (!$res) {
            return $this->_setError('FAILED_UPDATE_GRAPHICS', $this->graphics_id);
        }

        //$ret = $this->_updateImage();
        //var_dump($ret);


        $this->_clearError();
        return $res;
    }

    /**
     * 画像データを削除する
     */
    public function deleteGraphics()
    {
        global $wpdb;

        $res = $wpdb->delete(
            $this->tableName,
            array('graphics_id' => $this->graphics_id),
            array('%d')
        );

        if (!$res){
            return $this->_setError('FAILED_DELETE_GRAPHICS', $this->graphics_id);
        }

        return $res;
    }

    /**
     * 画像データのレンタルデータとのリレーション情報を更新する
     */
    public static function updateRentalId($graphicsId, $rentalId)
    {
        global $wpdb;

        // 更新保存日時
        $updated = current_time('mysql');

        $res = $wpdb->update(
            Config::getTableName(self::MODEL),
            array(
                'rental_id' => $rentalId,
                'updated' => $updated,
            ),
            array('graphics_id' => $graphicsId),
            array('%d', '%s'),
            array('%d')
        );

        return $res;
    }

    // エラーをセットする
    private function _setError($errCode, $errSub='')
    {
        $this->errCode = $errCode;
        $this->errSub = $errSub;

        return false;
    }

    // エラーをクリアする
    private function _clearError()
    {
        $this->errCode = $errSub = '';
    }

    /**
     * データを戻す
     */
    public function getData()
    {
        return $this->data;
    }

    /**
     * データをオブジェクトにセットする
     */
    public function setData($data)
    {
        $this->data = $data;
    }

    /**
     * プロパティから読み出す
     */
    public function __get($key)
    {
        if (array_key_exists($key, $this->data)) {
            if (is_array($this->data[$key]) && $key !== 'table') {
                return (object) $this->data[$key];
            }
            return $this->data[$key];
        }

        if (isset($this->$key)) {
            return $this->$key;
        }

        $trace = debug_backtrace();
        trigger_error(sprintf(
            "Undefined property: '%s&' in %s on line %d, E_USER_NOTICE",
            $key, $trace[0]['file'], $trace[0]['line']
        ));

        return null;
    }

    /**
     * プロパティをセットする
     */
    public function __set($key, $value)
    {
        if (array_key_exists($key, $this->data)) {
            if (is_array($value)) {
                foreach ($value as $item => $val) {
                    $this->data[$key][$item] = $val;
                }
            } else {
                $this->data[$key] = $value;
            }
        } else {
            $this->$key = $value;
        }

        return $value;
    }

    /**
     * 画像テーブルの生成とアップデート
     */
    public static function installTable(&$version, $list)
    {
        // スケジュールテーブル名を取得する
        $tableName = Config::getTableName(self::MODEL);

        // テーブルが登録されていない、またはバージョンが異なれば更新する
        if (array_key_exists(self::MODEL, $version) && in_array($tableName, $list)) {
            if ($version[self::MODEL] === self::VERSION) {
                return false;
            }
        }

        $sql = "CREATE TABLE $tableName (
            graphics_id int(11) unsigned NOT NULL AUTO_INCREMENT,
            rental_id int(11) DEFAULT '0',
            type varchar(32) DEFAULT '',
            filename varchar(128) DEFAULT '',
            extension varchar(16) DEFAULT '',
            image longblob,
            updated datetime DEFAULT NULL,
            created datetime DEFAULT NULL,
            PRIMARY KEY  (graphics_id),
            KEY rental_id (rental_id),
            KEY type (type)
        ) DEFAULT CHARSET=utf8mb4;";

        dbDelta($sql);

        // バージョンデータの更新
        $version[self::MODEL] = self::VERSION;

        return true;
    }

    public function phpError($errVal)
    {
        static $err = array(
            0 => 'UPLOAD_ERR_OK',
            1 => 'UPLOAD_ERR_INI_SIZE',
            2 => 'UPLOAD_ERR_FORM_SIZE',
            3 => 'UPLOAD_ERR_PARTIAL',
            4 => 'UPLOAD_ERR_NO_FILE',
            6 => 'UPLOAD_ERR_CANT_WRITE',
            7 => 'UPLOAD_ERR_EXTENSION',
        );

        if (array_key_exists($errVal, $err)) {
            return $err[$errVal];
        }

        return 'ERROR';
    }

    /**
     * 画像データ出力
     */
    public static function image($graphicsId)
    {
        global $wpdb;

        $sql = $wpdb->prepare("SELECT * FROM " . Config::getTableName(self::MODEL)
            . " WHERE graphics_id=%d", $graphicsId);

        $data = $wpdb->get_row($sql, ARRAY_A);

        if ($data) {
            header(sprintf("Content-Type: image/%s", $data['extension']));
            echo $data['image'];
        }

        exit();
    }

}
