<?php
/**
 * MTS Car Booking カスタマーモデル
 *
 * @Filename    Customer.php
 * @Author      S.Hayashi
 * @Code        2018-06-28 Ver.1.0.0
 */
namespace MTSCarBookingTrial\models;

use MTSCarBookingTrial\Config;

class Customer
{
    // Customer 入力項目指定
    const DISCARDED = 0;
    const AVAILABLE = 1;
    const REQUIRED = 2;
    const PHONE = 3;

    // カスタマーデータ
    private $data = array(
        'company' => '',    // 会社・団体名
        'sei' => '',        // 姓
        'mei' => '',        // 名
        'sei_kana' => '',   // 姓読み
        'mei_kana' => '',   // 名読み
        'email' => '',      // Eメール
        'email2' => '',     // Eメール確認
        'postcode' => '',   // 〒番号
        'pref' => '',       // 都道府県名
        'city' => '',       // 郡市区名
        'town' => '',       // 町村番地
        'building' => '',   // アパートビル号
        'country' => '',    // 国
        'tel' => '',        // 電話番号(携帯番号かどちらか一方)
        'mobile' => '',     // 携帯電話番号
        'fax' => '',        // FAX番号
        'gender' => '',     // 性別
        'birthday' => '',   // 誕生日
        'lang' => Config::LANG,       // 言語指定
        'external' => array(),  // 外部拡張
    );

    // エラー
    public $error = array();

    // 共通鍵
/*
    private function _getKey($key)
    {
        return substr($key . constant('AUTH_KEY'), 0, 64);
    }
*/
    /**
     * 保存用暗号化データを取得する
     */
    public function getEncryptData($key)
    {
        return openssl_encrypt(serialize($this->data), 'AES-256-ECB', $key);
        //return openssl_encrypt(serialize($this->data), 'AES-256-ECB', $this->_getKey($key));
    }

    /**
     * 保存暗号化データを復元してプロパティに展開する
     */
    public function setDecryptData($encrypted, $key)
    {
        $data = unserialize(openssl_decrypt($encrypted, 'AES-256-ECB', $key));
        //$data = unserialize(openssl_decrypt($encrypted, 'AES-256-ECB', $this->_getKey($key)));

        if ($data) {
            $this->setData($data);
            return true;
        }

        return false;
    }

    /**
     * 設定データを渡して利用するデータカラム名配列を戻す
     */
    public function getUseColumn($setting=array())
    {
        if (empty($setting)) {
            return array_keys($this->data);
        }

        $columns = array();

        foreach ($setting as $key => $usage) {
            if (0 < $usage) {
                switch ($key) {
                    case 'seimei':
                        $columns = array_merge($columns, array(
                            'sei' => $usage, 'mei' => $usage));
                        break;
                    case 'furigana':
                        $columns = array_merge($columns, array(
                            'sei_kana' => $usage, 'mei_kana' => $usage));
                        break;
                    case 'jusho':
                        $columns = array_merge($columns, array(
                            'pref' => $usage, 'city' => $usage, 'town' => $usage, 'building' => $usage));
                        break;
                    case 'company':
                    case 'email':
                    case 'email2':
                    case 'postcode':
                    case 'country':
                    case 'tel':
                    case 'mobile':
                    case 'gender':
                    case 'birthday':
                        $columns[$key] = $usage;
                        break;
                    default:
                        break;
                }
            }
        }

        return $columns;
    }

    /**
     * 個人情報入力処理
     */
    public function normalize($columns, $post)
    {
        // 入力データを正規化して取得する
        foreach ($columns as $key => $usage) {
            if (isset($post[$key])) {
                switch ($key) {
                    case 'company':
                    case 'sei':
                    case 'mei':
                    case 'email':
                    case 'postcode':
                    case 'pref':
                    case 'city':
                    case 'town':
                    case 'building':
                    case 'country':
                    case 'tel':
                    case 'mobile':
                    case 'fax':
                    case 'gender':
                    case 'lang':
                        $val = trim(mb_convert_kana($post[$key], 'as'));
                        break;
                    case 'sei_kana':
                    case 'mei_kana':
                        $val = trim(mb_convert_kana($post[$key], 'sCKV'));
                        break;
                    case 'birthday':
                        $val = sprintf('%04d-%02d-%02d',
                            $post['birthday']['year'], $post['birthday']['month'], $post['birthday']['day']);
                        break;
                    case 'email2':
                        $val = trim($post[$key]);
                        break;
                }

                $this->$key = $val;
            }
        }

    }

    /**
     * 必須入力項目とバリデーションのチェックを実行する
     */
    public function checkValidation($columns)
    {
        // 入力データをチェックする
        foreach ($columns as $key => $usage) {
            // 未使用項目はスキップする
            if ($usage <= self::DISCARDED) {
                continue;
            }

            // 必須入力項目をチェックする
            if ($usage == self::REQUIRED) {
                if ($this->data[$key] == '') {
                    $errKey = $key;

                    if ($key == 'sei' || $key == 'mei') {
                        $errKey = 'seimei';
                    } elseif ($key == 'sei_kana' || $key == 'mei_kana') {
                        $errKey = 'furigana';
                    } elseif (in_array($key, array('pref', 'city', 'town', 'building'))) {
                        // buildingは除外する
                        if ($key != 'building') {
                            $errKey = 'jusho';
                        }
                    }
                    $this->setError($errKey, 'ERROR_REQUIRED');
                }
            }

            if ($key == 'postcode') {
                if (!$this->checkPostcode($this->postcode)) {
                    $this->setError($key, 'INVALID_POSTCODE');
                }
            }
            elseif ($key == 'tel' or $key == 'mobile' or $key == 'fax') {
                if (!$this->checkTelno($this->$key)) {
                    $this->setError($key, 'INVALID_PHONE_NUMBER');
                }
            }
        }

        // メールチェック
        if ($columns['email'] == self::REQUIRED && $columns['email2'] == self::REQUIRED) {
            if ($this->email != $this->email2) {
                $this->setError('email2', 'UNMATCHED_EMAIL');
            }
        }

        // 電話番号チェック
        if ($columns['tel'] == self::PHONE && $columns['mobile'] == self::PHONE) {
            if ($this->tel == '' && $this->mobile == '') {
                $this->setError('mobile', 'NO_NUMBER_PHONE');
            }
        }

        return empty($this->error) ? true : false;
    }

    // 郵便番号の確からしさ確認
    public function checkPostcode($postcode)
    {
        if ($postcode == '' || preg_match('/^[0-9\-]*$/', $postcode)) {
            return true;
        }

        return false;
    }

    // 電話番号の確からしさ確認
    public function checkTelno($tel)
    {
        if (empty($tel) || preg_match("/^[0-9\-\(\)]*$/", $tel)) {
            return true;
        }

        return false;
    }

    /**
     * データを配列で一括取得する
     */
    public function getData()
    {
        return $this->data;
    }

    /**
     * データを一括セットする
     */
    public function setData($data)
    {
        foreach ($data as $key => $val) {
            if (isset($this->data[$key])) {
                $this->data[$key] = $val;
            }
        }
    }

    /**
     * エラーコードを取得する
     */
    public function getError($itemName)
    {
        if (isset($this->error[$itemName])) {
            return $this->error[$itemName];
        }

        return '';
    }

    /**
     * エラーをセットする
     */
    public function setError($key, $err)
    {
        $this->error[$key] = $err;

        return false;
    }

    /**
     * プロパティから読み出す
     */
    public function __get($key)
    {
        if (array_key_exists($key, $this->data)) {
            return $this->data[$key];
        }

        if (isset($this->$key)) {
            return $this->$key;
        }

        $trace = debug_backtrace();
        trigger_error(sprintf(
            "Undefined property: '%s&' in %s on line %d, E_USER_NOTICE",
            $key, $trace[0]['file'], $trace[0]['line']
        ));

        return null;
    }

    /**
     * プロパティをセットする
     */
    public function __set($key, $value)
    {
        if (array_key_exists($key, $this->data)) {
            $this->data[$key] = $value;
        } else {
            $this->$key = $value;
        }

        return $value;
    }

    /**
     * プロパティの確認
     */
    public function __isset($key)
    {
        return isset($this->data[$key]);
    }

}
