<?php
/**
 * MTS Car Booking レンタル料金管理
 *
 * @Filename    PaymentAdmin.php
 * @Author      S.Hayashi
 * @Code        2018-11-19 Ver.1.0.0
 */
namespace MTSCarBookingTrial;

use MTSCarBookingTrial\models\Rental;
use MTSCarBookingTrial\models\Vehicle;
use MTSCarBookingTrial\views\PaymentAdminView;

class PaymentAdmin
{
    const PAGE_NAME = CarBooking::ADMIN_PAYMENT;

    public $oRental = null;
    public $oVehicle = null;

    public $msgCode = '';
    public $msgSub = '';
    public $errflg = false;

    public $setting = array();

    // View
    private $view = NULL;

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->setting['payment'] = Config::getSetting('payment');
        $this->setting['customer'] = Config::getSetting('customer');

        $this->view = new PaymentAdminView($this);
    }

    /**
     * 料金処理
     */
    public function payment()
    {
        // レンタルデータ準備
        $this->oRental = new Rental;
        $rentalId = 0;

        if (isset($_POST['payment_nonce']) && wp_verify_nonce($_POST['payment_nonce'], self::PAGE_NAME)) {

            // レンタルデータを検索する
            if (isset($_POST['find_rental'])) {
                $this->oRental->reserve_id = $_POST['rental']['reserve_id'];
                $rentalId = Rental::findByReserveId($this->oRental->reserve_id, false);
            } else {
                $rentalId = intval($_POST['rental']['rental_id']);
            }

            // レンタル予約データを読込む
            if ($this->_getRentalData($rentalId)) {
                // 決済状況を更新する
                if (isset($_POST['update_accounting'])) {
                    $this->_updateRentalAccounting();
                } // 入力の料金計算をする
                elseif (isset($_POST['input_charge'])) {
                    $this->_inputRentalCharge();
                } // 初期の料金計算をする
                elseif (isset($_POST['init_charge'])) {
                    $this->oRental->charging($this->oVehicle);
                } // 料金計算の入力結果を保存する
                elseif (isset($_POST['save_charge'])) {
                    $this->_saveRentalCharge();
                } // 燃料代計算の入力結果を保存する
                elseif (isset($_POST['save_fuel'])) {
                    $this->_saveRentalFuel();
                } // キャンセル料の入力
                elseif (isset($_POST['input_cancel'])) {
                    $this->_inputCancelCharge();
                } // 現在時刻におけるキャンセル料を計算する
                elseif (isset($_POST['init_cancel'])) {
                    $this->oRental->cancelCharge();
                } // キャンセル料を保存する
                elseif (isset($_POST['save_cancel'])) {
                    $this->_saveCancelCharge();
                }

            } else {
                $this->_setError('NOT_FOUND', $this->oRental->rental_id);
            }
        }

        // レンタルデータIDの指定
        elseif (isset($_GET['rid'])) {
            $rentalId = intval($_GET['rid']);

            // 指定のレンタル予約データを読込む
            $this->_getRentalData($rentalId);
        }

        if ($this->oRental->rental_id <= 0) {
            if ($this->errflg) {
                $this->_setError('NOT_FOUND', $rentalId);
            }
            $this->view->findView();
        } else {
            $this->view->paymentView();
        }
    }

    // レンタルデータと車両データを取得する
    private function _getRentalData($rentalId=0)
    {
        // 指定のレンタル予約データを読込む
        if (!$this->oRental->getRental($rentalId)) {
            return $this->_setError($this->oRental->errCode, $this->oRental->errSub);
        }

        // 車両データを取得する
        $this->oVehicle = Vehicle::readVehicle($this->oRental->oVehicleSchedule->vehicle_id);

        return true;
    }

    // 清算状態を更新する
    private function _updateRentalAccounting()
    {
        $accounting = intval($_POST['rental']['accounting']);
        if ($this->oRental->updateAccounting($accounting)) {
            $this->_setMessage('SAVED_NORMALLY');
        } else {
            $this->_setError($this->oRental->errCode, $this->oRental->errSub);
        }
    }

    // 料金計算の入力結果を保存する
    private function _saveRentalCharge()
    {
        $this->_inputRentalCharge();

        if ($this->_updateCharge()) {
            $this->_setMessage('SAVED_NORMALLY');
        }
    }

    // 入力料金を取得する
    private function _inputRentalCharge()
    {
        $post = stripslashes_deep($_POST['payment']);

        // 車両スケジュールデータに料金をセットする
        $charge = $this->_strToNumber($post[$this->oRental->oVehicleSchedule->schedule_id]);
        $this->oRental->oVehicleSchedule->charge = $charge;

        // オプションスケジュールデータに料金をセットする
        foreach ($this->oRental->optionSchedules as $scheduleId => $oSchedule) {
            if ($oSchedule->select) {
                $oSchedule->charge = $this->_strToNumber($post[$oSchedule->schedule_id]);
            }
        }

        // レンタル料金の合計計算を実行する
        $this->oRental->calculateCharge();
    }

    private function _strToNumber($val)
    {
        $str = mb_convert_kana($val, 'as');

        return preg_replace('/[^0-9\-\.]/', '', $str);
    }

    // 料金計算を保存する
    private function _updateCharge()
    {
        // 車両スケジュールを更新する
        if (!$this->oRental->oVehicleSchedule->updateCharge()) {
            return $this->_setError($this->oRental->oVehicleSchedule->errCode, $this->oRental->oVehicleSchedule->errSub);
        }

        // オプションスケジュールを更新する
        foreach ($this->oRental->optionSchedules as $scheduleId => $oSchedule) {
            if (!$oSchedule->updateCharge()) {
                return $this->_setError($oSchedule->errCode, $oSchedule->errSub);
            }
        }

        if (!$this->oRental->updatePayment()) {
            return $this->_setError($this->oRental->errCode, $this->oRental->errSub);
        }

        return true;
    }

    // 燃料代計算の入力結果を保存する
    private function _saveRentalFuel()
    {
        // レンタルID、走行メーター、燃料代計算データを入力する
        $rentalId = intval($_POST['rental']['rental_id']);
        $odoStart = intval(mb_convert_kana(trim($_POST['schedule']['odo_start'], 'as')));
        $odoEnd = intval(mb_convert_kana(trim($_POST['schedule']['odo_end'])));
        $fuelPayment = $this->_strToNumber($_POST['payment']['fuel']);
        $fuelMileage = $this->_strToNumber($_POST['payment']['mileage']);

        // レタンルデータを取得する
        if ($this->_getRentalData($rentalId)) {

            // 走行メーターの値を更新する
            if ($this->oRental->oVehicleSchedule->updateOdo($odoStart, $odoEnd)) {
                $this->oRental->setPayment('fuel', $fuelPayment);
                $this->oRental->setPayment('mileage', $fuelMileage);

                // 燃料代を更新する
                if (!$this->oRental->updatePayment()) {
                    $this->_setError($this->oRental->errCode, $this->oRental->errSub);
                } else {
                    $this->_setMessage('SAVED_NORMALLY');
                }
            } else {
                $this->_setError($this->oRental->oVehicleSchedule->errCode, $this->oRental->oVehicleSchedule->errSub);
            }
        }

    }

    // キャンセル料を入力計算する
    private function _inputCancelCharge()
    {
        $settingPayment = Config::getSetting('payment');

        $cancel = $this->oRental->getPayment('cancel');

        if ($cancel == '') {
            $cancel = array(
                'total' => 0,
                'cancelCharge' => 0,
                'taxRate' => $settingPayment->tax,
                'tax' => 0,
                'consumption' => $settingPayment->consumption,
                'cancelTotal' => 0,
                'currency' => $settingPayment->currency,
                'cancelTime' => 0,
            );
        }

        $cancelCharge = $this->_strToNumber($_POST['cancel']['charge']);
        $tax = $this->oRental->computeTax($cancelCharge);
        $cancelTotal = $cancelCharge + (0 < $settingPayment->consumption ? $tax : 0);

        $cancel['cancelCharge'] = $cancelCharge;
        $cancel['tax'] = $tax;
        $cancel['cancelTotal'] = $cancelTotal;
        $cancel['cancelTime'] = current_time('timestamp');

        $this->oRental->setPayment('cancel', $cancel);
    }

    // キャンセル料を入力保存する
    private function _saveCancelCharge()
    {
        $this->_inputCancelCharge();

        if (!$this->oRental->updatePayment()) {
            $this->_setError($this->oRental->errCode, $this->oRental->errSub);
        } else {
            $this->_setMessage('SAVED_NORMALLY');
        }
    }

    // エラーをセットする
    private function _setError($errCode, $errSub='')
    {
        $this->msgCode = $errCode;
        $this->msgSub = $errSub;
        $this->errflg = true;

        return false;
    }

    // メッセージをセットする
    private function _setMessage($msgCode, $msgSub='')
    {
        $this->msgCode = $msgCode;
        $this->msgSub = $msgSub;

        return true;
    }

}
