<?php
/**
 * MTS Car Booking フロントエンド予約日付入力処理
 *
 * @Filename    BookingCar.php
 * @Author      S.Hayashi
 * @Code        2018-09-30 Ver.1.0.0
 */
namespace MTSCarBookingTrial;

use MTSCarBookingTrial\models\Calendar;
use MTSCarBookingTrial\models\ShopCalendar;
use MTSCarBookingTrial\models\Rental;
use MTSCarBookingTrial\models\Vehicle;
use MTSCarBookingTrial\models\Schedule;
use MTSCarBookingTrial\models\Mail;
use MTSCarBookingTrial\models\Alternate;
use MTSCarBookingTrial\lang\Translation;
use MTSCarBookingTrial\views\BookingCarView;

class BookingCar
{
    const PAGE_NAME = CarBooking::BOOKING_PAGE;

    const ENTRY_PAGE = CarBooking::BOOKING_ENYTY_PAGE;      // 日付入力表示ページ名
    const FORM_PAGE = CarBooking::BOOKING_FORM_PAGE;        // フォーム入力表示ページ名
    const ORDER_PAGE = CarBooking::BOOKING_ORDER_PAGE;      // 確認表示ページ名
    const ABORT_PAGE = CarBooking::BOOKING_ABORT_PAGE;      // 作業中止表示ページ
    const FINISH_PAGE = CarBooking::BOOKING_FINISH_PAGE;    // 予約終了ページ

    // セッションオブジェクト
    public $oSession = null;

    // 表示使用言語
    public $lang = Config::LANG;

    // 各種設定
    public $settings = array();

    // メッセージ
    public $msgCode = '';
    public $msgSub = '';
    public $errflg = false;

    // カレンダー表示データ定義
    private static $daily = array(
        'status' => 'closed',
        'classes' => array(),
        'memo' => '',
    );

    // 各種設定
    private $settingReserve = null;
    private $settingForm = null;
    private $settingCustomer = null;

    // カレンダーデータ
    private $oCalendar = null;

    // 予約受付期間
    private $acceptClose = 0;               // この日まで(直近)
    private $acceptOpen = 0;                // この日から(将来)

    // 車両データ
    private $oVehicle = null;

    // レンタルデータ
    private $oRental = null;

    private $shopCalendar = array();        // 指定月の営業カレンダー

    // 予約データシンボル
    private $oAlternate = null;

    // アップロードファイルデータ($_FILES)
    private $files = array();

    private $view = null;

    /**
     * Constructor
     */
    public function __construct()
    {
        // 各種設定の予約条件を取得する
        $this->settingReserve = Config::getSetting('reserve');
        $this->settingForm = Config::getSetting('form');
        $this->settingCustomer = Config::getSetting('customer');

        // 予約カレンダーを準備する
        $this->oCalendar = $this->_initCalendar($this->settingReserve);
    }


    // 予約カレンダーを準備する
    private function _initCalendar($setting)
    {
        $oCalendar = new Calendar;

        // 直近の予約受付有効日
        $this->acceptClose = $oCalendar->todayTime + $setting->close * 86400;

        // 将来の予約受付有効日
        $this->acceptOpen =  $oCalendar->todayTime + $setting->open * 86400;

        // 前月リンク有効期限
        $oCalendar->linkLast = $oCalendar->todayMonth;

        // 翌月リンク有効期限
        $futureYmd = explode('-', date('Y-n-j', $this->acceptOpen));
        $oCalendar->linkFuture = mktime(0, 0, 0, $futureYmd[1], 1, $futureYmd[0]);

        return $oCalendar;
    }


    /**
     * 予約日付入力処理
     */
    public function bookingEntry()
    {
        global $mts_car_booking_trial;

        $page = self::ENTRY_PAGE;

        // セッションオブジェクトをセットする
        $this->oSession = $mts_car_booking_trial->startSession();
        $this->lang = $this->oSession->lang;

        // レンタルデータを準備する
        $this->oRental = $this->_initRental();

        if (isset($_POST['rental_nonce']) && wp_verify_nonce($_POST['rental_nonce'], self::PAGE_NAME)) {
            // 入力データ
            $post = stripslashes_deep($_POST);

            // 決済処理ボタンが押されたら予約データを読み込む
            if (isset($post['reserve_payment'])) {
                $this->oRental->getRental(intval($post['rental']['rental_id']));
            }

            // それ以外はセッションデータを読込む
            else {
                $this->oRental->restoreSession($this->oSession);
            }

            // 車両データを取得する
            if (!$this->_readVehicle($this->oRental->oVehicleSchedule->vehicle_id)) {
                //$this->_setError('INVALID_OPERATION');
                $page = self::ABORT_PAGE;
                $post = array();
            }

            // レンタル期日の入力
            if (isset($post['new_rental'])) {
                if ($this->_checkEntry($post)) {
                    // レタンルデータを(再)構築する
                    $this->_prepareResource($this->oRental);
                    // レンタルデータをセッションデータに保存する
                    $this->oRental->saveSession($this->oSession);

                    $page = self::FORM_PAGE;
                }
            }

            // カレンダー日付入力へ戻る
            elseif (isset($post['return_entry'])) {
                // レンタルフォームを入力する
                $this->_inputForm($post);
                // セッションデータを保存する
                $this->oRental->saveSession($this->oSession);
            }

            // レンタルフォームの入力
            elseif (isset($post['check_form'])) {
                $page = self::FORM_PAGE;

                // レンタルフォームの入力処理
                if ($this->_inputForm($post)) {
                    // 入力(申込者)データをチェックする
                    if ($this->_inputCheck()) {
                        // 料金計算表示なら料金計算する
                        if (0 < $this->settingForm->charge) {
                            $this->oRental->charging($this->oVehicle);
                        }

                        $page = self::ORDER_PAGE;
                    }
                }

                $this->oRental->saveSession($this->oSession);
            }

            // レンタル予約を実行する
            elseif (isset($post['booking_rental'])) {
                if ($this->_bookingRental($this->oRental)) {
                    $this->oSession->clear(Rental::MODEL);
                    $this->_bookingMail();
                    $page = self::FINISH_PAGE;
                } else {
                    $page = self::ORDER_PAGE;
                }
            }

            // レンタルフォーム入力へ戻る
            elseif (isset($post['return_form'])) {
                $page = self::FORM_PAGE;
            }

            // 予約決済処理実行のエラー表示
            elseif (isset($post['reserve_payment'])) {
                // 予約データシンボル化
                $oAlternate = new Alternate($this->oRental->lang);
                $vars = $oAlternate->makeVar($this->oRental, $this->oVehicle);

                // 決済ページへ移行する
                $ret = $mts_car_booking_trial->redirectPayment($this->oRental, $vars);

                // 決済ページへ移行しない場合はエラーを設定する
                $this->_setError('NOT_INSTALLED_PAYMENT');
                $page = self::ABORT_PAGE;
            }

            // 予約処理を中止する
            elseif (isset($post['abort_rental'])) {
                // アップロード画像を削除する
                $this->oRental->abortRental();

                // セッションデータをクリアする
                $this->oSession->clear(Rental::MODEL);

                $page = self::ABORT_PAGE;
            }
        }

        // カレンダー表示レンタル予約日付入力表示開始
        else {
            // セッションをクリアする
            $this->oSession->clear(Rental::MODEL);

            // 車両IDを取得、確認する
            $vehicleId = isset($_GET['vid']) ? intval($_GET['vid']) : 0; //$this->oRental->oVehicleSchedule->vehicle_id;
            if ($this->_readVehicle($vehicleId)) {
                // 車両IDをセットする
                $this->oRental->oVehicleSchedule->vehicle_id = $vehicleId;

                $this->oRental->saveSession($this->oSession);

            } else {
                $page = self::ABORT_PAGE;
            }
        }

        // 予約新規登録・編集のビュー
        $this->view = new BookingCarView($this);

        return $page;
    }

    // 初期化したレンタルデータのオブジェクトを戻す
    private function _initRental()
    {
        $oRental = new Rental;

        // 仮レンタル開始日時 = 受付終了日＋開店時間
        $oRental->rent_start = $this->acceptClose + $this->settingReserve->opening;

        // 仮レンタル返却日時 = 受付終了日＋閉店時間
        $oRental->rent_end = $this->acceptClose + $this->settingReserve->closing;

        // ログインユーザーID
        $oRental->user_id = get_current_user_id();

        // 表示言語
        $oRental->lang = $this->lang;
        $oRental->oCustomer->lang = $this->lang;

        // 車両スケジュール用データを用意する
        $oRental->oVehicleSchedule = new Schedule;

        return $oRental;
    }

    // 車両データを読み込む
    private function _readVehicle($vehicleId)
    {
        $this->oVehicle = Vehicle::readVehicle($vehicleId);

        if ($this->oVehicle->postId <= 0) {
            return $this->_setError('CAR_NOT_FOUND', $vehicleId);
        }

        return true;
    }

    // 車両と予約日付の入力を確認する
    private function _checkEntry($post)
    {
        $inputRental = $post['rental'];
        //$inputSchedule = $post['schedule'];

        // 貸渡日・返却日UnixTime
        $rentStart = strtotime(date_i18n('Y-m-d', (int) $inputRental['start']['date']))
            + $inputRental['start']['hour'] * 3600 + $inputRental['start']['minute'] * 60;
        $rentEnd = strtotime(date_i18n('Y-m-d', (int) $inputRental['end']['date']))
            + $inputRental['end']['hour'] * 3600 + $inputRental['end']['minute'] * 60;

        // 期日転倒の確認
        if ($rentStart > $rentEnd) {
            list($rentEnd, $rentStart) = [$rentStart, $rentEnd];
        }

        // レンタル期間の入力日時
        $this->oRental->rent_start = $rentStart;
        $this->oRental->rent_end = $rentEnd;

        $startTime = $rentStart % 86400;
        $endTime = $rentEnd % 86400;
        $rentStart -= $startTime;
        $rentEnd -= $endTime;

        // 予約対象車両スケジュールデータにセットする
        $this->oRental->setSchedule($this->oRental->oVehicleSchedule);

        // 受付期間内か確認する
        if (!$this->_checkAccepting($rentStart) || !$this->_checkAccepting($rentEnd)) {
            return $this->_setError('OUTSIDE_ACCEPTED');
        }

        // 営業時間内か貸渡返却時間チェック
        if (!$this->_checkShopHours($startTime) || !$this->_checkShopHours($endTime)) {
            return $this->_setError('OUTSIDE_HOURS');
        }

        // 貸渡日と返却日の休業日と季節営業チェック
        if (!$this->_checkShopDate($rentStart) || !$this->_checkShopDate($rentEnd)) {
            return $this->_setError('SHOP_CLOSED');
        }

        // レンタル可能期間を超えていないか確認する
        if (!$this->_checkPeriod($rentStart, $rentEnd)) {
            return $this->_setError('TOO_LONG_DAYS', $this->settingReserve->limit);
        }

        // レンタル期間中の貸出チェック
        if (!$this->oRental->oVehicleSchedule->vehicleAvailable()) {
            return $this->_setError('CAR_BOOKED');
        }

        return true;
    }

    // 予約受付期間内か確認する
    private function _checkAccepting($dayTime)
    {
        if ($this->acceptClose <= $dayTime && $dayTime <= $this->acceptOpen) {
            return true;
        }

        return false;
    }

    // 営業時間内か確認する
    private function _checkShopHours($time)
    {
        if ($this->settingReserve->opening <= $time && $time <= $this->settingReserve->closing) {
            return true;
        }

        return false;
    }

    // 日付が休業日でないか確認する
    private function _checkShopDate($daytime)
    {
        if (!isset($this->shopCalendar[$daytime])) {
            // 貸渡日月の営業カレンダーを取得する
            $this->oCalendar->setCalendar($daytime);
            if (!$this->_getShopCalendar()) {
                return false;
            }
        }

        // 休業日か確認する
        if (!$this->shopCalendar[$daytime]['open']) {
            return false;
        }

        // 季節営業期間
        $on_time = strtotime(sprintf('%s-%s', $this->oCalendar->year, $this->settingReserve->on_season));
        $off_time = strtotime(sprintf('%s-%s', $this->oCalendar->year, $this->settingReserve->off_season));

        // 季節営業期間外か確認する
        if ($on_time < $off_time) {
            // 夏営業チェック
            if ($daytime < $on_time || $off_time < $daytime) {
                return false;
            }
        } elseif ($on_time > $off_time) {
            // 冬営業チェック
            if ($on_time < $daytime && $daytime < $off_time) {
                return false;
            }
        }

        return true;
    }

    // レンタル可能期間を超えていないか確認する
    private function _checkPeriod($rentStart, $rentEnd)
    {
        $days = ceil(($rentEnd - $rentStart) / 86400) + 1;

        if ($this->settingReserve->limit < $days) {
            return false;
        }

        return true;
    }


    // 車両データ、オプションデータと各スケジュールデータを準備する
    private function _prepareResource(Rental $oRental)
    {
        // 車両スケジュールデータを準備する
        if (empty($oRental->oVehicleSchedule)) {
            $oRental->oVehicleSchedule = $oRental->newSchedule($this->oVehicle->postId);
        } else {
            $oRental->oVehicleSchedule->vehicle_id = $this->oVehicle->postId;
            $oRental->setSchedule($oRental->oVehicleSchedule);
        }

        // オプションスケジュールデータを準備する
        $schedules = array();
        foreach ($this->oVehicle->options as $oOption) {
            // 編集中または新規オプションスケジュールデータを戻す
            $oSchedule = $this->_getSchedule($oOption->postId);
            $oRental->setSchedule($oSchedule);

            // 必須選択を設定する
            if ($oOption->type) {
                $oSchedule->select = true;
            }

            // 貸出可能在庫がなければ選択不可を設定する
            if (!$oSchedule->rentalAvailable($oOption)) {
                $oSchedule->unavailable = true;
            }

            $schedules[] = $oSchedule;
        }

        $oRental->optionSchedules = $schedules;
    }

    // 編集中または新規オプションスケジュールデータを戻す
    private function _getSchedule($optionId)
    {
        foreach ($this->oRental->optionSchedules as $oSchedule) {
            if ($optionId == $oSchedule->option_id) {
                return $oSchedule;
            }
        }

        return $this->oRental->newSchedule(0, $optionId);
    }

    // 予約フォームから入力する
    private function _inputForm($post)
    {
        // 選択オプション
        foreach ($this->oRental->optionSchedules as $oOptionSchedule) {
            if ($post['option']['schedule'][$oOptionSchedule->option_id] == 1) {
                $oOptionSchedule->select = true;
            } else {
                $oOptionSchedule->select = false;
            }
        }

        // 申込者フォームの入力項目の配列を取得する
        $columns = $this->oRental->oCustomer->getUseColumn($this->settingCustomer);

        // 申込者
        $this->oRental->oCustomer->normalize($columns, $post['customer']);

        // 画像アップロード情報
        if (!empty($_FILES)) {
            $files = stripslashes_deep($_FILES);

            if (isset($files['rental_passport'])) {
                $this->files['passport'] = $files['rental_passport'];
            }

            if (isset($files['rental_license'])) {
                $this->files['license'] = $files['rental_license'];
            }
        }

        return true;
    }

    // 入力(申込者)データをチェックする
    private function _inputCheck()
    {
        $ret = true;

        // 申込者フォームの入力項目の配列を取得する
        $columns = $this->oRental->oCustomer->getUseColumn($this->settingCustomer);

        // 申込者入力データをチェックする
        if (!$this->oRental->oCustomer->checkValidation($columns)) {
            $ret = $this->_setError('INVALID_CUSTOMER_INPUT');
        }

        // パスポート画像アップロードを保存、チェックする
        if (0 < $this->settingForm->passport) {
            if (!$this->_checkUpload('passport')) {
                $ret = false;
            }
        }

        // 免許証画像アップロードを保存、チェックする
        if (0 < $this->settingForm->license) {
            if (!$this->_checkUpload('license')) {
                $ret = false;
            }
        }

        return $ret;
    }

    // アップロード画像をチェック、データベースへ登録する
    private function _checkUpload($type)
    {
        // アップロード済み　または　アップロードが任意のとき
        if (0 < $this->oRental->upload[$type] || $this->settingForm->{$type} <= 1) {
            // ファイルアップロードないときOK
            if (!isset($this->files[$type]) || $this->files[$type]['error'] == 4) {
                return true;
            }
        }

        // アップロード必須
        if (2 <= $this->settingForm->{$type}) {
            // ファイルアップロードがないときNG
            if (!isset($this->files[$type]) || $this->files[$type]['error'] == 4) {
                return $this->_setError('IMAGE_NOT_UPLOAD', Translation::$bForm['FORM_' . strtoupper($type)][$this->lang]);
            }
        }

        // アップロードされたテンポラリデータをデータベースへ登録する
        if (!$this->oRental->saveGraphics($type, $this->files[$type])) {
            return $this->_setError($this->oRental->errCode, $this->oRental->errSub);
        }

        return true;
    }

    // レンタル予約を実行する
    private function _bookingRental(Rental $oRental)
    {
        // 規約同意チェックを確認する
        if (0 < $this->settingForm->concent) {
            $acceptance = intval($_POST['rental']['acceptance']);
            if (1 < $this->settingForm->concent && $acceptance <= 0) {
                return $this->_setError('TERMS_CHECK_MISSING');
            }
            $oRental->acceptance = $acceptance;
        }

        // 車両のレンタル状況確認
        if (!$oRental->oVehicleSchedule->vehicleAvailable()) {
            return $this->_setError('CAR_BOOKED_ALREADY');
        }

        // オプションのレンタル状況確認
        foreach ($oRental->optionSchedules as $oOptionSchedule) {
            if ($oOptionSchedule->select) {
                if (!$oOptionSchedule->rentalAvailable($this->oVehicle->options[$oOptionSchedule->option_id])) {
                    return $this->_setError('OPTION_BOOKED_ALREADY');
                }
            }
        }

        // 料金計算する
        if (0 < $this->settingForm->charge) {
            $oRental->charging($this->oVehicle);
        }

        // レタンルデータを新規登録する
        if (!$oRental->newRental()) {
            return $this->_setError($this->oRental->errCode, $this->oRental->errSub);
        }

        return true;
    }

    // 予約メールを送信する
    private function _bookingMail()
    {
        if ($this->settingForm->mail <= 0) {
            return true;
        }

        // メールテンプレート
        $oMail = Mail::readTemplate($this->settingForm->mail, $this->oRental->lang);
        if ($oMail->postId <= 0) {
            return $this->_setError('TEMPLATE_NOT_FOUND', $this->settingForm->mail);
        }

        // 予約データをシンボル化する
        $this->oAlternate = new Alternate($this->oRental->lang);
        $this->oAlternate->makeVar($this->oRental, $this->oVehicle);

        // メール送信
        if (!$oMail->sendMail($this->oAlternate)) {
            return $this->_setError($oMail->errCode, $oMail->errSub);
        }

        return true;
    }


    /**
     * 予約日時入力
     */
    public function entryDate()
    {
        // 予約の受付中止
        if (!$this->settingReserve->accept) {
            $this->_setError('BOOKING_CLOSED');
            return $this->view->abortPage();
        }

        // 表示月のカレンダーオブジェクト設定
        $this->oCalendar->setCalendar($this->oRental->rent_start);

        $oBookingCalendar = new BookingCalendar($this->oCalendar);
        $calendar = $oBookingCalendar->calendar($this->oVehicle->postId, $this->oCalendar->todayMonth, $this->lang);

        if ($calendar == '') {
            $this->_setError($oBookingCalendar->msgCode, $oBookingCalendar->msgSub);
            return $this->view->abortPage();
        }

        return $this->view->entryForm($this->oRental, $this->oVehicle, $this->oCalendar, $calendar);
    }

    // 指定月の営業カレンダーを取得する
    private function _getShopCalendar()
    {
        $oShopCalendar = new ShopCalendar;

        if (!$oShopCalendar->readShopCalendar($this->oCalendar->monthTime)) {
            return $this->_setError('SCHEDULE_NOT_DETERMINED');
        }

        $shopMonth =  $oShopCalendar->getShopMonth($this->oCalendar->monthTime);

        $shopCalendar = array();

        // 営業カレンダー１カ月に展開する
        $datetime = $this->oCalendar->monthTime;
        foreach ($shopMonth as $shopData) {
            $shopCalendar[$datetime] = $shopData;
            $datetime += 86400;
        }

        $this->shopCalendar = $shopCalendar;

        return true;
    }

    // エラーセット
    private function _setError($code, $sub='')
    {
        $this->msgCode = $code;
        $this->msgSub = $sub;
        $this->errflg = true;
        return false;
    }

    /**
     * 予約フォーム入力
     */
    public function entryForm()
    {
        return $this->view->rentalForm($this->oRental, $this->oVehicle);
    }

    /**
     * 予約発注フォーム入力
     */
    public function orderForm()
    {
        return $this->view->rentalOrder($this->oRental, $this->oVehicle);
    }

    /**
     * 予約を中止する
     */
    public function abortRental()
    {
        // エラーによる中止
        if ($this->errflg) {
            $messageId = 'HOME_RETURN';
            $linkUrl = get_home_url();
        }

        // 中止指示
        else {
            $messageId = 'ABORT_RETURN';
            $linkUrl = add_query_arg(array(
                'vid' => $this->oRental->oVehicleSchedule->vehicle_id,
                'lang' => $this->oRental->lang,
            ), get_permalink(get_page_by_path(CarBooking::BOOKING_PAGE)));

            $this->_setError('ABORT_BOOKING');
        }

        return $this->view->abortPage($messageId, $linkUrl);
    }

    /**
     * 予約処理終了
     */
    public function finishBooking()
    {
        // 予約データをシンボル化する
        if (is_null($this->oAlternate)) {
            $this->oAlternate = new Alternate($this->oRental->lang);
            $this->oAlternate->makeVar($this->oRental, $this->oVehicle);
        }

        return $this->view->finishPage($this->oRental, $this->oVehicle, $this->oAlternate);
    }

    /**
     * AJAX予約カレンダー切替
     */
    public function changeCalendar($post)
    {
        $lang = $post['lang'];

        // 予約の受付中止
        if (!$this->settingReserve->accept) {
            $this->_setError('BOOKING_CLOSED');
            return $this->_errorMessage();
        }

        // 車両IDを取得、確認する
        $vehicleId = $post['vehicle_id'];
        if (!$this->_readVehicle($vehicleId)) {
            return $this->_errorMessage();
        }

        // カレンダー表示月
        $monthTime = $post['month'];

        // 指定月が表示期間内かチェックする
        if ($monthTime < $this->oCalendar->linkLast) {
            $monthTime = $this->oCalendar->linkLast;
        } elseif ($this->oCalendar->linkFuture < $monthTime) {
            $monthTime = $this->oCalendar->linkFuture;
        }

        // 表示月のカレンダーオブジェクト設定
        $this->oCalendar->setCalendar($monthTime);

        $oBookingCalendar = new BookingCalendar($this->oCalendar);
        $calendar = $oBookingCalendar->calendar($vehicleId, $this->oCalendar->monthTime, $lang);

        if ($calendar == '') {
            $this->_setError($oBookingCalendar->msgCode, $oBookingCalendar->msgSub);
            return $this->_errorMessage();
        }

        return $calendar;
    }

    // エラーメッセージを戻す
    private function _errorMessage()
    {
        if (!$this->errflg) {
            return '';
        }

        if (isset(Translation::$fMessage[$this->msgCode])) {
            return sprintf(Translation::$fMessage[$this->msgCode][$this->lang], $this->msgSub);
        }

        return $this->msgCode . ($this->msgSub == '' ? '' : " : {$this->msgSub}");
    }

}
